-- the debugger 2.1 Standalone 

dcsCommon = {}
dcsCommon.version = "3.0.1"
--[[-- VERSION HISTORY 
3.0.0  - removed bad bug in stringStartsWith, only relevant if caseSensitive is false 
       - point2text new intsOnly option 
	   - arrangeGroupDataIntoFormation minDist harden
	   - cleanup 
	   - new pointInDirectionOfPointXYY()
	   - createGroundGroupWithUnits now supports liveries
	   - new getAllExistingPlayersAndUnits()
3.0.1  - clone: better handling of string type 
--]]--

	-- dcsCommon is a library of common lua functions 
	-- for easy access and simple mission programming
	-- (c) 2021 - 2023 by Chritian Franz and cf/x AG

	dcsCommon.verbose = false -- set to true to see debug messages. Lots of them
	dcsCommon.uuidStr = "uuid-"
	dcsCommon.simpleUUID = 76543 -- a number to start. as good as any
	
	-- globals
	dcsCommon.cbID = 0 -- callback id for simple callback scheduling
	dcsCommon.troopCarriers = {"Mi-8MT", "UH-1H", "Mi-24P"} -- Ka-50, Apache and Gazelle can't carry troops
	dcsCommon.coalitionSides = {0, 1, 2}
	dcsCommon.maxCountry = 86 -- number of countries defined in total 
	
	-- lookup tables
	dcsCommon.groupID2Name = {}
	dcsCommon.unitID2Name = {}
	dcsCommon.unitID2X = {}
	dcsCommon.unitID2Y = {}

	-- verify that a module is loaded. obviously not required
	-- for dcsCommon, but all higher-order modules
	function dcsCommon.libCheck(testingFor, requiredLibs)
		local canRun = true 
		for idx, libName in pairs(requiredLibs) do 
			if not _G[libName] then 
				trigger.action.outText("*** " .. testingFor .. " requires " .. libName, 30)
				canRun = false 
			end
		end
		return canRun
	end

	-- read all groups and units from miz and build a reference table
	function dcsCommon.collectMissionIDs()
	-- create cross reference tables to be able to get a group or
	-- unit's name by ID
		for coa_name_miz, coa_data in pairs(env.mission.coalition) do -- iterate all coalitions
			local coa_name = coa_name_miz
			if string.lower(coa_name_miz) == 'neutrals' then -- remove 's' at neutralS
				coa_name = 'neutral'
			end
			-- directly convert coalition into number for easier access later
			local coaNum = 0
			if coa_name == "red" then coaNum = 1 end 
			if coa_name == "blue" then coaNum = 2 end 
			
			if type(coa_data) == 'table' then -- coalition = {bullseye, nav_points, name, county}, 
											  -- with county being an array 
				if coa_data.country then -- make sure there a country table for this coalition
					for cntry_id, cntry_data in pairs(coa_data.country) do -- iterate all countries for this 
						-- per country = {id, name, vehicle, helicopter, plane, ship, static}
						local countryName = string.lower(cntry_data.name)
						local countryID = cntry_data.id 
						if type(cntry_data) == 'table' then	-- filter strings .id and .name 
							for obj_type_name, obj_type_data in pairs(cntry_data) do
								-- only look at helos, ships, planes and vehicles
								if obj_type_name == "helicopter" or 
								   obj_type_name == "ship" or 
								   obj_type_name == "plane" or 
								   obj_type_name == "vehicle" or 
								   obj_type_name == "static" -- what about "cargo"?
								then -- (so it's not id or name)
									local category = obj_type_name
									if ((type(obj_type_data) == 'table') and obj_type_data.group and (type(obj_type_data.group) == 'table') and (#obj_type_data.group > 0)) then	--there's at least one group!
										for group_num, group_data in pairs(obj_type_data.group) do
											
											local aName = group_data.name 
											local aID = group_data.groupId
											-- store this reference 
											dcsCommon.groupID2Name[aID] = aName 
											
											-- now iterate all units in this group 
											-- for player into 
											for unit_num, unit_data in pairs(group_data.units) do
												if unit_data.name and unit_data.unitId then 
													-- store this reference 
													dcsCommon.unitID2Name[unit_data.unitId] = unit_data.name
													dcsCommon.unitID2X[unit_data.unitId] = unit_data.x
													dcsCommon.unitID2Y[unit_data.unitId] = unit_data.y
												end
											end -- for all units
										end -- for all groups 
									end --if has category data 
								end --if plane, helo etc... category
							end --for all objects in country 
						end --if has country data 
					end --for all countries in coalition
				end --if coalition has country table 
			end -- if there is coalition data  
		end --for all coalitions in mission 
	end

	function dcsCommon.getUnitNameByID(theID)
		-- accessor function for later expansion
		return dcsCommon.unitID2Name[theID]
	end
	
	function dcsCommon.getGroupNameByID(theID)
		-- accessor function for later expansion 
		return dcsCommon.groupID2Name[theID]
	end

	function dcsCommon.getUnitStartPosByID(theID)
		local x = dcsCommon.unitID2X[theID]
		local y = dcsCommon.unitID2Y[theID]
		return x, y
	end
	
	-- returns only positive values, lo must be >0 and <= hi 
	function dcsCommon.randomBetween(loBound, hiBound)
		if not loBound then loBound = 1 end 
		if not hiBound then hiBound = 1 end 
		if loBound == hiBound then return loBound end 

		local delayMin = loBound
		local delayMax = hiBound 
		local delay = delayMax 
	
		if delayMin ~= delayMax then 
			-- pick random in range , say 3-7 --> 5 s!
			local delayDiff = (delayMax - delayMin) + 1 -- 7-3 + 1
			delay = dcsCommon.smallRandom(delayDiff) - 1 --> 0-4
			delay = delay + delayMin 
			if delay > delayMax then delay = delayMax end 
			if delay < 1 then delay = 1 end 
		
			if dcsCommon.verbose then 
				trigger.action.outText("+++dcsC: delay range " .. delayMin .. "-" .. delayMax .. ": selected " .. delay, 30)
			end
		end
		
		return delay
	end
	

	-- taken inspiration from mist, as dcs lua has issues with
	-- random numbers smaller than 50. Given a range of x numbers 1..x, it is 
	-- repeated a number of times until it fills an array of at least 
	-- 50 items (usually some more), and only then one itemis picked from 
	-- that array with a random number that is from a greater range (0..50+)
	function dcsCommon.smallRandom(theNum) -- adapted from mist, only support ints
		theNum = math.floor(theNum)
		if theNum >= 50 then return math.random(theNum) end
		if theNum < 1 then
			trigger.action.outText("smallRandom: invoke with argument < 1 (" .. theNum .. "), using 1", 30)
			theNum = 1 
		end 
		-- for small randoms (<50) 
		local lowNum, highNum
		highNum = theNum
		lowNum = 1
		local total = 1
		if math.abs(highNum - lowNum + 1) < 50 then -- if total values is less than 50
			total = math.modf(50/math.abs(highNum - lowNum + 1)) -- number of times to repeat whole range to get above 50. e.g. 11 would be 5 times 1 .. 11, giving us 55 items total 
		end
		local choices = {}
		for i = 1, total do -- iterate required number of times
			for x = lowNum, highNum do -- iterate between the range
				choices[#choices +1] = x -- add each entry to a table
			end
		end
		local rtnVal; -- = math.random(#choices) -- will now do a math.random of at least 50 choices
		for i = 1, 15 do
			rtnVal = math.random(#choices) -- iterate 15 times for randomization
		end
		return choices[rtnVal] -- return indexed
	end
	

	function dcsCommon.getSizeOfTable(theTable)
		local count = 0
		for _ in pairs(theTable) do count = count + 1 end
		return count
	end

	function dcsCommon.findAndRemoveFromTable(theTable, theElement) -- assumes array 
		if not theElement then return false end 
		if not theTable then return false end 
		for i=1, #theTable do 
			if theTable[i] == theElement then 
				-- this element found. remove from table 
				table.remove(theTable, i)
				return true 
			end
		end
	end

	function dcsCommon.pickRandom(theTable)
		if not theTable then 
			trigger.action.outText("*** warning: nil table in pick random", 30)
		end
		
		if #theTable < 1 then 
			trigger.action.outText("*** warning: zero choice in pick random", 30)
			--local k = i.ll 
			return nil
		end
		if #theTable == 1 then return theTable[1] end
		r = dcsCommon.smallRandom(#theTable) --r = math.random(#theTable)
		return theTable[r]
	end

	-- enumerateTable - make an array out of a table for indexed access
	function dcsCommon.enumerateTable(theTable)
		if not theTable then theTable = {} end
		local array = {}
		for key, value in pairs(theTable) do 
			table.insert(array, value)
		end
		return array
	end

	-- combine table. creates new 
	function dcsCommon.combineTables(inOne, inTwo)
		local outTable = {}
		for idx, element in pairs(inOne) do 
			table.insert(outTable, element)
		end
		for idx, element in pairs(inTwo) do 
			table.insert(outTable, element)
		end
		return outTable
	end
	
	function dcsCommon.addToTableIfNew(theTable, theElement)
		for idx, anElement in pairs(theTable) do 
			if anElement == theElement then return end 
		end
		table.insert(theTable, theElement)
	end
-- 
-- A I R F I E L D S  A N D  F A R P S  
--

	-- airfield management 
	function dcsCommon.getAirbaseCat(aBase)
		if not aBase then return nil end 
		
		local airDesc = aBase:getDesc()
		if not airDesc then return nil end 
		
		local airCat = airDesc.category
		return airCat 
	end

	-- get free parking slot. optional parkingType can be used to 
	-- filter for a scpecific type, e.g. 104 = open field
	function dcsCommon.getFirstFreeParkingSlot(aerodrome, parkingType) 
		if not aerodrome then return nil end 
		local freeSlots = aerodrome:getParking(true)
		
		for idx, theSlot in pairs(freeSlots) do 
			if not parkingType then 
				-- simply return the first we come across
				return theSlot
			end		
			
			if theSlot.Term_Type == parkingType then 
				return theSlot 
			end
		end
		
		return nil 
	end

	-- getAirbasesInRangeOfPoint: get airbases that are in range of point 
	function dcsCommon.getAirbasesInRangeOfPoint(center, range, filterCat, filterCoalition)
		if not center then return {} end 
		if not range then range = 500 end -- 500m default 
		local basesInRange = {}
		
		local allAB = dcsCommon.getAirbasesWhoseNameContains("*", filterCat, filterCoalition)
		for idx, aBase in pairs(allAB) do 			
			local delta = dcsCommon.dist(center, aBase:getPoint())
			if delta <= range then 
				table.insert(basesInRange, aBase)
			end
		end
		return basesInRange
	end

	-- getAirbasesInRangeOfAirbase returns all airbases that 
	-- are in range of the given airbase 
	function dcsCommon.getAirbasesInRangeOfAirbase(airbase, includeCenter, range, filterCat, filterCoalition)
		if not airbase then return {} end
		if not range then range = 150000 end 
		local center = airbase:getPoint() 
		local centerName = airbase:getName() 
		
		local ABinRange = {}
		local allAB = dcsCommon.getAirbasesWhoseNameContains("*", filterCat, filterCoalition)
		
		for idx, aBase in pairs(allAB) do 
			if aBase:getName() ~= centerName then 
				local delta = dcsCommon.dist(center, aBase:getPoint())
				if delta <= range then 
					table.insert(ABinRange, aBase)
				end
			end		
		end
		
		if includeCenter then 
			table.insert(ABinRange, airbase)
		end
		
		return ABinRange
	end

	function dcsCommon.getAirbasesInRangeOfAirbaseList(theCenterList, includeList, range, filterCat, filterCoalition)
		local collectorDict = {}
		for idx, aCenter in pairs(theCenterList) do 
			-- get all surrounding airbases. returns list of airfields 
			local surroundingAB = dcsCommon.getAirbasesInRangeOfAirbase(airbase, includeList, range, filterCat, filterCoalition)
			
			for idx2, theAirField in pairs (surroundingAB) do 
				collectorDict[airField] = theAirField 
			end
		end
		
		-- make result an array
		local theABList = dcsCommon.enumerateTable(collectorDict)
		return theABList
	end

	-- getAirbasesWhoseNameContains - get all airbases containing 
	-- a name. filterCat is optional and can be aerodrome (0), farp (1), ship (2)
	-- filterCoalition is optional and can be 0 (neutral), 1 (red), 2 (blue) or 
	-- a table containing categories, e.g. {0, 2} = airfields and ships but not farps 
	-- if no name given or aName = "*", then all bases are returned prior to filtering 
	function dcsCommon.getAirbasesWhoseNameContains(aName, filterCat, filterCoalition)
		if not aName then aName = "*" end 
		local allYourBase = world.getAirbases() -- get em all 
		local areBelongToUs = {}
		-- now iterate all bases
		for idx, aBase in pairs(allYourBase) do
			local airBaseName = aBase:getName() -- get display name
			if aName == "*" or dcsCommon.containsString(airBaseName, aName) then 
				-- containsString is case insesitive unless told otherwise
				local doAdd = true  
				if filterCat then 
					local aCat = dcsCommon.getAirbaseCat(aBase)
					if type(filterCat) == "table" then 
						local hit = false
						for idx, fCat in pairs(filterCat) do 
							if fCat == aCat then hit = true end
						end
						doAdd = doAdd and hit 
					else 
						-- make sure the airbase is of that category 
						local airCat = aCat
						doAdd = doAdd and airCat == filterCat 
					end
				end
				
				if filterCoalition then 
					doAdd = doAdd and filterCoalition == aBase:getCoalition()
				end
				
				if doAdd then 
					-- all good, add to table
					table.insert(areBelongToUs, aBase)
				end			
			end
		end
		return areBelongToUs
	end

	function dcsCommon.getFirstAirbaseWhoseNameContains(aName, filterCat, filterCoalition)
		local allBases = dcsCommon.getAirbasesWhoseNameContains(aName, filterCat, filterCoalition)
		for idx, aBase in pairs (allBases) do 
			-- simply return first 
			return aBase
		end
		return nil 
	end	

	function dcsCommon.getClosestAirbaseTo(thePoint, filterCat, filterCoalition, allYourBase)
		local delta = math.huge
		if not allYourBase then 
			allYourBase = dcsCommon.getAirbasesWhoseNameContains("*", filterCat, filterCoalition) -- get em all and filter
		end 
		
		local closestBase = nil 
		for idx, aBase in pairs(allYourBase) do
			-- iterate them all 
			local abPoint = aBase:getPoint()
			newDelta = dcsCommon.dist(thePoint, {x=abPoint.x, y = 0, z=abPoint.z})
			if newDelta < delta then 
				delta = newDelta
				closestBase = aBase
			end
		end
		return closestBase, delta 
	end

	function dcsCommon.getClosestFreeSlotForCatInAirbaseTo(cat, x, y, theAirbase, ignore)
		if not theAirbase then return nil end 
		if not ignore then ignore = {} end 
		if not cat then return nil end 
		if (not cat == "helicopter") and (not cat == "plane") then 
			trigger.action.outText("+++common-getslotforcat: wrong cat <" .. cat .. ">", 30)
			return nil 
		end
		local allFree = theAirbase:getParking(true) --  only free slots
		local filterFreeByType = {}
		for idx, aSlot in pairs(allFree) do 
			local termT = aSlot.Term_Type
			if termT == 104 or 
			(termT == 72 and cat == "plane") or 
			(termT == 68 and cat == "plane") or 
			(termT == 40 and cat == "helicopter") then 
				table.insert(filterFreeByType, aSlot)
			else 
				-- we skip this slot, not good for type 
			end
		end
		
		if #filterFreeByType == 0 then 
			return nil
		end 
		
		local reallyFree = {}
		for idx, aSlot in pairs(filterFreeByType) do 
			local slotNum = aSlot.Term_Index
			isTaken = false 
			for idy, taken in pairs(ignore) do 
				if taken == slotNum then isTaken = true end 
			end
			if not isTaken then 
				table.insert(reallyFree, aSlot)
			end
		end
		
		if #reallyFree < 1 then 
			reallyFree = filterFreeByType
		end
		
		local closestDist = math.huge 
		local closestSlot = nil 
		local p = {x = x, y = 0, z = y} -- !!
		for idx, aSlot in pairs(reallyFree) do 
			local sp = {x = aSlot.vTerminalPos.x, y = 0, z = aSlot.vTerminalPos.z}
			local currDist = dcsCommon.distFlat(p, sp)
			if currDist < closestDist then 
				closestSlot = aSlot 
				closestDist = currDist 
			end
		end
		return closestSlot
	end

-- 
-- U N I T S   M A N A G E M E N T 
--

	-- number of living units in group
	function dcsCommon.livingUnitsInGroup(group)
		local living = 0
		local allUnits = group:getUnits()
		for key, aUnit in pairs(allUnits) do 
			if aUnit:isExist() and aUnit:getLife() >= 1 then 
				living = living + 1
			end
		end
		return living
	end

	-- closest living unit in group to a point
	function dcsCommon.getClosestLivingUnitToPoint(group, p)
		if not p then return nil end
		if not group then return nil end
		local closestUnit = nil
		local closestDist = math.huge
		local allUnits = group:getUnits()
		for key, aUnit in pairs(allUnits) do 
			if aUnit:isExist() and aUnit:getLife() >= 1 then 
				local thisDist = dcsCommon.dist(p, aUnit:getPoint())
				if thisDist < closestDist then 
					closestDist = thisDist
					closestUnit = aUnit 
				end
			end
		end
		return closestUnit, closestDist
	end
	
	-- closest living group to a point - cat can be nil or one of Group.Category = { AIRPLANE = 0, HELICOPTER = 1, GROUND = 2, SHIP = 3, TRAIN = 4}
	function dcsCommon.getClosestLivingGroupToPoint(p, coal, cat) 
		if not cat then cat = 2 end -- ground is default 
		local closestGroup = nil;
		local closestGroupDist = math.huge
		local allGroups =  coalition.getGroups(coal, cat) -- get all groups from this coalition, perhaps filtered by cat 
		for key, grp in pairs(allGroups) do
			local closestUnit, dist = dcsCommon.getClosestLivingUnitToPoint(grp, p)
			if closestUnit then 
				if dist < closestGroupDist then 
					closestGroup = grp
					closestGroupDist = dist
				end
			end			
		end
		return closestGroup, closestGroupDist
	end

	function dcsCommon.getLivingGroupsAndDistInRangeToPoint(p, range, coal, cat) 
		if not cat then cat = 2 end -- ground is default 
		local groupsInRange = {};
		local allGroups = coalition.getGroups(coal, cat) -- get all groups from this coalition, perhaps filtered by cat 
		for key, grp in pairs(allGroups) do
			local closestUnit, dist = dcsCommon.getClosestLivingUnitToPoint(grp, p)
			if closestUnit then 
				if dist < range then 
					table.insert(groupsInRange, {group = grp, dist = dist}) -- array
				end
			end			
		end
		-- sort the groups by distance
		table.sort(groupsInRange, function (left, right) return left.dist < right.dist end )
		return groupsInRange
	end

	-- distFlat ignores y, input must be xyz points, NOT xy points  
	function dcsCommon.distFlat(p1, p2) 
		local point1 = {x = p1.x, y = 0, z=p1.z}
		local point2 = {x = p2.x, y = 0, z=p2.z}
		return dcsCommon.dist(point1, point2)
	end
	
	
	-- distance between points
	function dcsCommon.dist(point1, point2)	 -- returns distance between two points
	  -- supports xyz and xy notations
	  if not point1 then 
		trigger.action.outText("+++ warning: nil point1 in common:dist", 30)
		point1 = {x=0, y=0, z=0}
	  end

	  if not point2 then 
		trigger.action.outText("+++ warning: nil point2 in common:dist", 30)
		point2 = {x=0, y=0, z=0}
		stop.here.now = 1
	  end
	  
	  local p1 = {x = point1.x, y = point1.y}
	  if not point1.z then 
		p1.z = p1.y
		p1.y = 0
	  else 
		p1.z = point1.z
	  end
	  
	  local p2 = {x = point2.x, y = point2.y}
	  if not point2.z then 
		p2.z = p2.y
		p2.y = 0
	  else 
		p2.z = point2.z
	  end
	  
	  local x = p1.x - p2.x
	  local y = p1.y - p2.y 
	  local z = p1.z - p2.z
	  
	  return (x*x + y*y + z*z)^0.5
	end

	function dcsCommon.delta(name1, name2) -- returns distance (in meters) of two named objects
	  local n1Pos = Unit.getByName(name1):getPosition().p
	  local n2Pos = Unit.getByName(name2):getPosition().p
	  return dcsCommon.dist(n1Pos, n2Pos)
	end

	-- lerp between a and b, x being 0..1 (percentage), clipped to [0..1]
	function dcsCommon.lerp(a, b, x) 
		if not a then return 0 end
		if not b then return 0 end
		if not x then return a end
		if x < 0 then x = 0 end 
		if x > 1 then x = 1 end 
		return a + (b - a ) * x
	end

	function dcsCommon.bearingFromAtoB(A, B) -- coords in x, z 
		if not A then 
			trigger.action.outText("WARNING: no 'A' in bearingFromAtoB", 30)
			return 0
		end
		if not B then
			trigger.action.outText("WARNING: no 'B' in bearingFromAtoB", 30)
			return 0
		end
		if not A.x then 
			trigger.action.outText("WARNING: no 'A.x' (type A =<" .. type(A) .. ">)in bearingFromAtoB", 30)
			return 0
		end
		if not A.z then 
			trigger.action.outText("WARNING: no 'A.z' (type A =<" .. type(A) .. ">)in bearingFromAtoB", 30)
			return 0
		end
		if not B.x then 
			trigger.action.outText("WARNING: no 'B.x' (type B =<" .. type(B) .. ">)in bearingFromAtoB", 30)
			return 0
		end
		if not B.z then 
			trigger.action.outText("WARNING: no 'B.z' (type B =<" .. type(B) .. ">)in bearingFromAtoB", 30)
			return 0
		end
		
		local dx = B.x - A.x
		local dz = B.z - A.z
		local bearing = math.atan2(dz, dx) -- in radiants
		return bearing
	end

	function dcsCommon.bearingFromAtoBusingXY(A, B) -- coords in x, y 
		if not A then 
			trigger.action.outText("WARNING: no 'A' in bearingFromAtoBXY", 30)
			return 0
		end
		if not B then
			trigger.action.outText("WARNING: no 'B' in bearingFromAtoBXY", 30)
			return 0
		end
		if not A.x then 
			trigger.action.outText("WARNING: no 'A.x' (type A =<" .. type(A) .. ">)in bearingFromAtoBXY", 30)
			return 0
		end
		if not A.y then 
			trigger.action.outText("WARNING: no 'A.y' (type A =<" .. type(A) .. ">)in bearingFromAtoBXY", 30)
			return 0
		end
		if not B.x then 
			trigger.action.outText("WARNING: no 'B.x' (type B =<" .. type(B) .. ">)in bearingFromAtoBXY", 30)
			return 0
		end
		if not B.y then 
			trigger.action.outText("WARNING: no 'B.y' (type B =<" .. type(B) .. ">)in bearingFromAtoBXY", 30)
			return 0
		end
		
		local dx = B.x - A.x
		local dz = B.y - A.y
		local bearing = math.atan2(dz, dx) -- in radiants
		return bearing
	end

	function dcsCommon.bearingInDegreesFromAtoB(A, B)
		local bearing = dcsCommon.bearingFromAtoB(A, B) -- in rads 
		bearing = math.floor(bearing / math.pi * 180)
		if bearing < 0 then bearing = bearing + 360 end
		if bearing > 360 then bearing = bearing - 360 end
		return bearing
	end
	
	function dcsCommon.compassPositionOfARelativeToB(A, B)
		-- warning: is REVERSE in order for bearing, returns a string like 'Sorth', 'Southwest'
		if not A then return "***error:A***" end
		if not B then return "***error:B***" end
		local bearing = dcsCommon.bearingInDegreesFromAtoB(B, A) -- returns 0..360
		if bearing < 23 then return "North" end 
		if bearing < 68 then return "NE" end
		if bearing < 112 then return "East" end 
		if bearing < 158 then return "SE" end 
		if bearing < 202 then return "South" end 
		if bearing < 248 then return "SW" end 
		if bearing < 292 then return "West" end
		if bearing < 338 then return "NW" end 
		return "North"
	end
	
	function dcsCommon.bearing2degrees(inRad)
		local degrees = inRad / math.pi * 180
		if degrees < 0 then degrees = degrees + 360 end 
		if degrees > 360 then degrees = degrees - 360 end 
		return degrees 
	end
	
	function dcsCommon.bearing2compass(inrad)
		local bearing = math.floor(inrad / math.pi * 180)
		if bearing < 0 then bearing = bearing + 360 end
		if bearing > 360 then bearing = bearing - 360 end
		return dcsCommon.bearingdegrees2compass(bearing)
	end
	
	function dcsCommon.bearingdegrees2compass(bearing)
		if bearing < 23 then return "North" end 
		if bearing < 68 then return "NE" end
		if bearing < 112 then return "East" end 
		if bearing < 158 then return "SE" end 
		if bearing < 202 then return "South" end 
		if bearing < 248 then return "SW" end 
		if bearing < 292 then return "West" end
		if bearing < 338 then return "NW" end 
		return "North"
	end
	
	function dcsCommon.clockPositionOfARelativeToB(A, B, headingOfBInDegrees)
		-- o'clock notation 
		if not A then return "***error:A***" end
		if not B then return "***error:B***" end
		if not headingOfBInDegrees then headingOfBInDegrees = 0 end 
		local bearing = dcsCommon.bearingInDegreesFromAtoB(B, A) -- returns 0..360 
		bearing = bearing - headingOfBInDegrees
		return dcsCommon.getClockDirection(bearing)
	end 
	
	-- given a heading, return clock with 0 being 12, 180 being 6 etc.
	function dcsCommon.getClockDirection(direction) -- inspired by cws, improvements my own
		if not direction then return 0 end
		direction = math.fmod (direction, 360)
		while direction < 0 do 
			direction = direction + 360
		end
		while direction >= 360 do 
			direction = direction - 360
		end
		if direction < 15 then -- special case 12 o'clock past 12 o'clock
			return 12
		end
	
		direction = direction + 15 -- add offset so we get all other times correct
		return math.floor(direction/30)
	
	end

	function dcsCommon.getGeneralDirection(direction) -- inspired by cws, improvements my own
		if not direction then return "unkown" end
		direction = math.fmod (direction, 360)
		while direction < 0 do 
			direction = direction + 360
		end
		while direction >= 360 do 
			direction = direction - 360
		end
		if direction < 45 then return "ahead" end	
		if direction < 135 then return "right" end
		if direction < 225 then return "behind" end
		if direction < 315 then return "left" end 
		return "ahead"
	end
	
	function dcsCommon.getNauticalDirection(direction) -- inspired by cws, improvements my own
		if not direction then return "unkown" end
		direction = math.fmod (direction, 360)
		while direction < 0 do 
			direction = direction + 360
		end
		while direction >= 360 do 
			direction = direction - 360
		end
		if direction < 45 then return "ahead" end	
		if direction < 135 then return "starboard" end
		if direction < 225 then return "aft" end
		if direction < 315 then return "port" end 
		return "ahead"
	end

	function dcsCommon.aspectByDirection(direction) -- inspired by cws, improvements my own
		if not direction then return "unkown" end
		direction = math.fmod (direction, 360)
		while direction < 0 do 
			direction = direction + 360
		end
		while direction >= 360 do 
			direction = direction - 360
		end
		
		if direction < 45 then return "hot" end	
		if direction < 135 then return "beam" end
		if direction < 225 then return "drag" end
		if direction < 315 then return "beam" end 
		return "hot"
	end
	
	function dcsCommon.whichSideOfMine(theUnit, target) -- returs two values: -1/1 = left/right and "left"/"right" 
		if not theUnit then return nil end 
		if not target then return nil end 
		local uDOF = theUnit:getPosition() -- returns p, x, y, z Vec3
		-- with x, y, z being the normalised vectors for right, up, forward 
		local heading = math.atan2(uDOF.x.z, uDOF.x.x) -- returns rads
		if heading < 0 then
			heading = heading + 2 * math.pi	-- put heading in range of 0 to 2*pi
		end
		-- heading now runs from 0 through 2Pi
		local A = uDOF.p
		local B = target:getPoint() 
		 
		-- now get bearing from theUnit to target  
		local dx = B.x - A.x
		local dz = B.z - A.z
		local bearing = math.atan2(dz, dx) -- in rads
		if bearing < 0 then
			bearing = bearing + 2 * math.pi	-- make bearing 0 to 2*pi
		end

		-- we now have bearing to B, and own heading. 
		-- subtract own heading from bearing to see at what 
		-- bearing target would be if we 'turned the world' so
		-- that theUnit is heading 0
		local dBearing = bearing - heading
		-- if result < 0 or > Pi (=180°), target is left from us
		if dBearing < 0 or dBearing > math.pi then return -1, "left" end
		return 1, "right"
		-- note: no separate case for straight in front or behind
	end
	
	-- Distance of point p to line defined by p1,p2 
	-- only on XZ map 
	function dcsCommon.distanceOfPointPToLineXZ(p, p1, p2)
		local x21 = p2.x - p1.x 
		local y10 = p1.z - p.z 
		local x10 = p1.x - p.x 
		local y21 = p2.z - p1.z 
		local numer = math.abs((x21*y10) - (x10 * y21))
		local denom = math.sqrt(x21 * x21 + y21 * y21)
		local dist = numer/denom 
		return dist 
	end
	
	function dcsCommon.randomDegrees()
		local degrees = math.random(360) * 3.14152 / 180
		return degrees
	end

	function dcsCommon.randomPercent()
		local percent = math.random(100)/100
		return percent
	end

	function dcsCommon.randomPointOnPerimeter(sourceRadius, x, z) 
		return dcsCommon.randomPointInCircle(sourceRadius, sourceRadius-1, x, z)
	end

	function dcsCommon.randomPointInCircle(sourceRadius, innerRadius, x, z)
		if not x then x = 0 end
		if not z then z = 0 end 
		
		--local y = 0
		if not innerRadius then innerRadius = 0 end		
		if innerRadius < 0 then innerRadius = 0 end
		
		local percent = dcsCommon.randomPercent() -- 1 / math.random(100)
		-- now lets get a random degree
		local degrees = dcsCommon.randomDegrees() -- math.random(360) * 3.14152 / 180 -- ok, it's actually radiants. 
		local r = (sourceRadius-innerRadius) * percent 
		x = x + (innerRadius + r) * math.cos(degrees)
		z = z + (innerRadius + r) * math.sin(degrees)
	
		local thePoint = {}
		thePoint.x = x
		thePoint.y = 0
		thePoint.z = z 
		
		return thePoint, degrees
	end

	function dcsCommon.newPointAtDegreesRange(p1, degrees, radius)
		local rads = degrees * 3.14152 / 180
		local p2 = dcsCommon.newPointAtAngleRange(p1, rads, radius)
		return p2 
	end
	
	function dcsCommon.newPointAtAngleRange(p1, angle, radius)
		local p2 = {}
		p2.x = p1.x + radius * math.cos(angle)
		p2.y = p1.y 
		p2.z = p1.z + radius * math.sin(angle)
		return p2 
	end

	-- get group location: get the group's location by 
	-- accessing the fist existing, alive member of the group that it finds
	function dcsCommon.getGroupLocation(group)
		-- nifty trick from mist: make this work with group and group name
		if type(group) == 'string' then -- group name
			group = Group.getByName(group)
		end
		
		-- get all units
		local allUnits = group:getUnits()

		-- iterate through all members of group until one is alive and exists
		for index, theUnit in pairs(allUnits) do 
			if (theUnit:isExist() and theUnit:getLife() > 0) then 
				return theUnit:getPosition().p 
			end;
		end

		-- if we get here, there was no live unit 
		return nil 
		
	end

	-- get the group's first Unit that exists and is 
	-- alive 
	function dcsCommon.getGroupUnit(group)
		if not group then return nil  end
		
		-- nifty trick from mist: make this work with group and group name
		if type(group) == 'string' then -- group name
			group = Group.getByName(group)
		end
		
		if not group:isExist() then return nil end 
		
		-- get all units
		local allUnits = group:getUnits()

		-- iterate through all members of group until one is alive and exists
		for index, theUnit in pairs(allUnits) do 
			if Unit.isExist(theUnit) and theUnit:getLife() > 0 then 
				return theUnit
			end;
		end

		-- if we get here, there was no live unit 
		return nil 
		
	end

	-- and here the alias
	function dcsCommon.getFirstLivingUnit(group)
		return dcsCommon.getGroupUnit(group)
	end
	
	-- isGroupAlive returns true if there is at least one unit in the group that isn't dead
	function dcsCommon.isGroupAlive(group)
		return (dcsCommon.getGroupUnit(group) ~= nil) 
	end

	function dcsCommon.getLiveGroupUnits(group)
		-- nifty trick from mist: make this work with group and group name
		if type(group) == 'string' then -- group name
			group = Group.getByName(group)
		end
		
		local liveUnits = {}
		-- get all units
		local allUnits = group:getUnits()

		-- iterate through all members of group until one is alive and exists
		for index, theUnit in pairs(allUnits) do 
			if (theUnit:isExist() and theUnit:getLife() > 0) then 
				table.insert(liveUnits, theUnit) 
			end;
		end

		-- if we get here, there was no live unit 
		return liveUnits
	end

	function dcsCommon.getGroupTypeString(group) -- convert into comma separated types 
		if not group then 
			trigger.action.outText("+++cmn getGroupTypeString: nil group", 30)
			return "" 
		end
		if not dcsCommon.isGroupAlive(group) then 
			trigger.action.outText("+++cmn getGroupTypeString: dead group", 30)
			return "" 
		end 
		local theTypes = ""
		local liveUnits = dcsCommon.getLiveGroupUnits(group)
		for i=1, #liveUnits do 
			if i > 1 then theTypes = theTypes .. "," end
			theTypes = theTypes .. liveUnits[i]:getTypeName()
		end
		return theTypes
	end

	function dcsCommon.getGroupTypes(group) 
		if not group then 
			trigger.action.outText("+++cmn getGroupTypes: nil group", 30)
			return {}
		end
		if not dcsCommon.isGroupAlive(group) then 
			trigger.action.outText("+++cmn getGroupTypes: dead group", 30)
			return {}
		end 
		local liveUnits = dcsCommon.getLiveGroupUnits(group)
		local unitTypes = {}
		for i=1, #liveUnits do 
			table.insert(unitTypes, liveUnits[i]:getTypeName())
		end
		return unitTypes
	end

	function dcsCommon.getEnemyCoalitionFor(aCoalition)
		if type(aCoalition) == "string" then 
			aCoalition = aCoalition:lower()
			if aCoalition == "red" then return 2 end
			if aCoalition == "blue" then return 1 end
			return nil 
		end
		if aCoalition == 1 then return 2 end
		if aCoalition == 2 then return 1 end
		return nil
	end

	function dcsCommon.getACountryForCoalition(aCoalition)
		-- scan the table of countries and get the first country that is part of aCoalition
		-- this is useful if you want to create troops for a coalition but don't know the
		-- coalition's countries 
		-- we start with id=0 (Russia), go to id=85 (Slovenia), but skip id = 14
		local i = 0
		while i < dcsCommon.maxCountry do -- 86 do 
			if i ~= 14 then 
				if (coalition.getCountryCoalition(i) == aCoalition) then return i end
			end
			i = i + 1
		end
		
		return nil
	end
	
	function dcsCommon.getCountriesForCoalition(aCoalition)
		if not aCoalition then aCoalition = 0 end 
		local allCty = {}
		
		local i = 0
		while i < dcsCommon.maxCountry do 
			if i ~= 14 then -- there is no county 14
				if (coalition.getCountryCoalition(i) == aCoalition) then 
					table.insert(allCty, i) 
				end
			end
			i = i + 1
		end
		return allCty
	end
--
--
-- C A L L B A C K   H A N D L E R 
--
--

	-- installing callbacks
	-- based on mist, with optional additional hooks for pre- and post-
	-- processing of the event
	-- when filtering occurs in pre, an alternative 'rejected' handler can be called 
	function dcsCommon.addEventHandler(f, pre, post, rejected) -- returns ID 
		local handler = {} -- build a wrapper and connect the onEvent
		handler.id = dcsCommon.uuid("eventHandler")
		handler.f = f -- the callback itself
		if (rejected) then handler.rejected = rejected end
		-- now set up pre- and post-processors. defaults are set in place
		-- so pre and post are optional. If pre returns false, the callback will
		-- not be invoked
		if (pre) then handler.pre = pre else handler.pre = dcsCommon.preCall end
		if (post) then handler.post = post else handler.post = dcsCommon.postCall end
		function handler:onEvent(event)
			if not self.pre(event) then 
				if dcsCommon.verbose then
				end
				if (self.rejected) then self.rejected(event) end 
				return
			end
			self.f(event) -- call the handler
			self.post(event) -- do post-processing
		end
		world.addEventHandler(handler)
		return handler.id
	end

	function dcsCommon.preCall(e)
		-- we can filter here
		-- if we return false, the call is abortet
		if dcsCommon.verbose then
			trigger.action.outText("event " .. e.id .. " received: PRE-PROCESSING", 10)
		end
		return true;
	end;

	function dcsCommon.postCall(e)
		-- we do pos proccing here 
		if dcsCommon.verbose then
			trigger.action.outText("event " .. e.id .. " received: post proc", 10)
		end
	end
	
	-- highly specific eventhandler for one event only
	-- based on above, with direct filtering built in; skips pre
	-- but does post
	function dcsCommon.addEventHandlerForEventTypes(f, evTypes, post, rejected) -- returns ID 
		local handler = {} -- build a wrapper and connect the onEvent
		dcsCommon.cbID = dcsCommon.cbID + 1 -- increment unique count
		handler.id = dcsCommon.cbID
		handler.what = evTypes
		if (rejected) then handler.rejected = rejected end 
		
		handler.f = f -- set the callback itself
		-- now set up post-processor. pre is hard-coded to match evType
		-- post is optional. If event.id is not in evTypes, the callback will
		-- not be invoked
		if (post) then handler.post = post else handler.post = dcsCommon.postCall end
		function handler:onEvent(event)
			hasMatch = false;
			for key, evType in pairs(self.what) do
				if evType == event.id then
					hasMatch = true;
					break;
				end;
			end;
			if not hasMatch then 
				if dcsCommon.verbose then
					trigger.action.outText("event " .. e.id .. " discarded - not in whitelist evTypes", 10)
				end
				if (self.rejected) then self.rejected(event) end 
				return;
			end;
			
			self.f(event) -- call the actual handler as passed to us
			self.post(event) -- do post-processing 
		end
		world.addEventHandler(handler) -- add to event handlers
		return handler.id
	end
	
	
	
	-- remove event handler / callback, identical to Mist 
	-- note we don't call world.removeEventHandler, but rather directly 
	-- access world.eventHandlers directly and remove kvp directly.
	function dcsCommon.removeEventHandler(id)
		for key, handler in pairs(world.eventHandlers) do
			if handler.id and handler.id == id then
				world.eventHandlers[key] = nil
				return true
			end
		end
		return false
	end

--
--
-- C L O N I N G 
--
--
	-- topClone is a shallow clone of orig, only top level is iterated,
	-- all values are ref-copied
	function dcsCommon.topClone(orig)
		if not orig then return nil end 
		local orig_type = type(orig)
		local copy
		if orig_type == 'table' then
			copy = {}
			for orig_key, orig_value in pairs(orig) do
				copy[orig_key] = orig_value
			end
		else -- number, string, boolean, etc
			copy = orig
		end
		return copy
	end

	-- clone is a recursive clone which will also clone
	-- deeper levels, as used in units 
	function dcsCommon.clone(orig, stripMeta)
		if not orig then return nil end 
		local orig_type = type(orig)
		local copy
		if orig_type == 'table' then
			copy = {}
			for orig_key, orig_value in next, orig, nil do
				copy[dcsCommon.clone(orig_key)] = dcsCommon.clone(orig_value)
			end
			if not stripMeta then 
				-- also connect meta data
				setmetatable(copy, dcsCommon.clone(getmetatable(orig)))
			else 
				-- strip all except string, and for strings use a fresh string 
				if type(copy) == "string" then 
					local tmp = ""
					tmp = tmp .. copy -- will get rid of any foreign metas for string 
					copy = tmp 
				end
			end
		elseif orig_type == "string" then 
			local tmp = ""
			copy = tmp .. orig 
		else -- number, string, boolean, etc
			copy = orig
		end
		return copy
	end

	function dcsCommon.copyArray(inArray)
		if not inArray then return nil end 
		
		-- warning: this is a ref copy!
		local theCopy = {}
		for idx, element in pairs(inArray) do 
			table.insert(theCopy, element)
		end
		return theCopy 
	end
--
-- 
-- S P A W N I N G 
-- 
-- 

	function dcsCommon.createEmptyGroundGroupData (name)
		local theGroup = {} -- empty group
		theGroup.visible = false
		theGroup.taskSelected = true
		-- theGroup.route = {}
		-- theGroup.groupId = id
		theGroup.tasks = {}
		-- theGroup.hidden = false -- hidden on f10?

		theGroup.units = { } -- insert units here! -- use addUnitToGroupData

		theGroup.x = 0
		theGroup.y = 0
		theGroup.name = name
		-- theGroup.start_time = 0
		theGroup.task = "Ground Nothing"
		
		return theGroup
	end;

	function dcsCommon.createEmptyAircraftGroupData (name)
		local theGroup = dcsCommon.createEmptyGroundGroupData(name)--{} -- empty group

		theGroup.task = "Nothing" -- can be others, like Transport, CAS, etc
		-- returns with empty route
		theGroup.route = dcsCommon.createEmptyAircraftRouteData() -- we can add points here 
		return theGroup
	end;

	function dcsCommon.createAircraftRoutePointData(x, z, altitudeInFeet, knots, altType, action)
		local rp = {}
		rp.x = x
		rp.y = z
		rp.action = "Turning Point"
		rp.type = "Turning Point"
		if action then rp.action = action; rp.type = action end -- warning: may not be correct, need to verify later
		rp.alt = altitudeInFeet * 0.3048 -- in m 
		rp.speed = knots * 0.514444 -- we use m/s
		rp.alt_type = "BARO"
		if (altType) then rp.alt_type = altType end 
		return rp
	end

	function dcsCommon.addRoutePointDataToRouteData(inRoute, x, z, altitudeInFeet, knots, altType, action)
		local p = dcsCommon.createAircraftRoutePointData(x, z, altitudeInFeet, knots, altType, action)
		local thePoints = inRoute.points 
		table.insert(thePoints, p)
	end
	
	function dcsCommon.addRoutePointDataToGroupData(group, x, z, altitudeInFeet, knots, altType, action)
		if not group.route then group.route = dcsCommon.createEmptyAircraftRouteData() end
		local theRoute = group.route 
		dcsCommon.addRoutePointDataToRouteData(theRoute, x, z, altitudeInFeet, knots, altType, action)
	end

	function dcsCommon.addRoutePointForGroupData(theGroup, theRP)
		if not theGroup then return end 
		if not theGroup.route then theGroup.route = dcsCommon.createEmptyAircraftRouteData() end
		
		local theRoute = theGroup.route 
		local thePoints = theRoute.points 
		table.insert(thePoints, theRP)
	end
	
	function dcsCommon.createEmptyAircraftRouteData()
		local route = {}
		route.points = {}
		return route
	end

	function dcsCommon.createTakeOffFromParkingRoutePointData(aerodrome)
		if not aerodrome then return nil end 
			
		local rp = {}	
		local freeParkingSlot = dcsCommon.getFirstFreeParkingSlot(aerodrome, 104) -- get big slot first 
		if not freeParkingSlot then 
			freeParkingSlot = dcsCommon.getFirstFreeParkingSlot(aerodrome) -- try any size
		end
			
		if not freeParkingSlot then 
			trigger.action.outText("civA: no free parking at " .. aerodrome:getName(), 30)
			return nil 
		end
			
		local p = freeParkingSlot.vTerminalPos
			
		rp.airdromeId = aerodrome:getID() 
		rp.x = p.x
		rp.y = p.z
		rp.alt = p.y 
		rp.action = "From Parking Area"
		rp.type = "TakeOffParking"
			
		rp.speed = 100 --  that's 360 km/h 
		rp.alt_type = "BARO"
		return rp
	end

	function dcsCommon.createOverheadAirdromeRoutePointData(aerodrome)
		if not aerodrome then return nil end 
		local rp = {}			
		local p = aerodrome:getPoint()
		rp.x = p.x
		rp.y = p.z
		rp.alt = p.y + 2000 -- 6000 ft overhead
		rp.action = "Turning Point"
		rp.type = "Turning Point"
			
		rp.speed = 133; -- in m/s? If so, that's 360 km/h 
		rp.alt_type = "BARO"
		return rp
	end
	function dcsCommon.createOverheadAirdromeRoutPintData(aerodrome) -- backwards-compat to typo 
		return dcsCommon.createOverheadAirdromeRoutePointData(aerodrome)
	end 
	

	function dcsCommon.createLandAtAerodromeRoutePointData(aerodrome)
		if not aerodrome then return nil end 
			
		local rp = {}			
		local p = aerodrome:getPoint()
		rp.airdromeId = aerodrome:getID() 
		rp.x = p.x
		rp.y = p.z
		rp.alt = land.getHeight({x=p.x, y=p.z}) --p.y 
		rp.action = "Landing"
		rp.type = "Land"
			
		rp.speed = 100; -- in m/s? If so, that's 360 km/h 
		rp.alt_type = "BARO"
		return rp
	end

	function dcsCommon.createSimpleRoutePointData(p, alt, speed)
		if not speed then speed = 133 end 
		if not alt then alt = 8000 end -- 24'000 feet 
		local rp = {}
		rp.x = p.x
		rp.y = p.z
		rp.alt = alt
		rp.action = "Turning Point"
		rp.type = "Turning Point"
			
		rp.speed = speed; -- in m/s? If so, that's 360 km/h 
		rp.alt_type = "BARO"
		return rp
	end 
	
	function dcsCommon.createRPFormationData(findex) -- must be added as "task" to an RP. use 4 for Echelon right
		local task = {}
		task.id = "ComboTask"
		local params = {}
		task.params = params
		local tasks = {}
		params.tasks = tasks
		local t1 = {}
		tasks[1] = t1
		t1.number = 1
		t1.auto = false 
		t1.id = "WrappedAction"
		t1.enabled = true
		local t1p = {}
		t1.params = t1p
		local action = {}
		t1p.action = action 
		action.id = "Option"
		local ap = {}
		action.params = ap
		ap.variantIndex = 3
		ap.name = 5 -- AI.Option.Air.ID 5 = Formation 
		ap.formationIndex = findex -- 4 is echelon_right
		ap.value = 262147
		
		return task 
	end

	function dcsCommon.addTaskDataToRP(theTask, theGroup, rpIndex)
		local theRoute = theGroup.route
		local thePoints = theRoute.points
		local rp = thePoints[rpIndex]
		rp.task = theTask
	end
	
	-- create a minimal payload table that is compatible with creating 
	-- a unit. you may need to alter this before adding the unit to
	-- the mission. all params optional 
	function dcsCommon.createPayload(fuel, flare, chaff, gun) 
		local payload = {}
		payload.pylons = {}
		if not fuel then fuel = 1000 end -- in kg. check against fuelMassMax in type desc
		if not flare then flare = 0 end
		if not chaff then chaff = 0 end
		if not gun then gun = 0 end
		return payload 
		
	end

	function dcsCommon.createCallsign(cs) 
		local callsign = {}
		callsign[1] = 1
		callsign[2] = 1
		callsign[3] = 1
		if not cs then cs = "Enfield11" end
		callsign.name = cs
		return callsign
	end
	

	-- create the data table required to spawn a unit.
	-- unit types are defined in https://github.com/mrSkortch/DCS-miscScripts/tree/master/ObjectDB
	function dcsCommon.createGroundUnitData(name, unitType, transportable)
		local theUnit = {}
		unitType = dcsCommon.trim(unitType)
		theUnit.type = unitType -- e.g. "LAV-25",
		if not transportable then transportable = false end -- elaborate, not requried code
		theUnit.transportable = {["randomTransportable"] = transportable} 
		-- theUnit.unitId = id 
		theUnit.skill = "Average" -- always average 
		theUnit.x = 0 -- make it zero, zero!
		theUnit.y = 0
		theUnit.name = name
		theUnit.playerCanDrive = false
		theUnit.heading = 0
		return theUnit
	end 

	function dcsCommon.createAircraftUnitData(name, unitType, transportable, altitude, speed, heading)
		local theAirUnit = dcsCommon.createGroundUnitData(name, unitType, transportable)
		theAirUnit.alt = 100 -- make it 100m
		if altitude then theAirUnit.alt = altitude end 
		theAirUnit.alt_type = "RADIO" -- AGL
		theAirUnit.speed = 77 -- m/s --> 150 knots
		if speed then theAirUnit.speed = speed end 
		if heading then theAirUnit.heading = heading end 
		theAirUnit.payload = dcsCommon.createPayload()
		theAirUnit.callsign = dcsCommon.createCallsign()
		return theAirUnit
	end
	

	function dcsCommon.addUnitToGroupData(theUnit, theGroup, dx, dy, heading)
		-- add a unit to a group, and place it at dx, dy of group's position,
		-- taking into account unit's own current location
		if not dx then dx = 0 end
		if not dy then dy = 0 end
		if not heading then heading = 0 end
		theUnit.x = theUnit.x + dx + theGroup.x
		theUnit.y = theUnit.y + dy + theGroup.y 
		theUnit.heading = heading
		table.insert(theGroup.units, theUnit)
	end;

	function dcsCommon.createSingleUnitGroup(name, theUnitType, x, z, heading) 
		-- create the container 
		local theNewGroup = dcsCommon.createEmptyGroundGroupData(name)
		local aUnit = {}
		aUnit = dcsCommon.createGroundUnitData(name .. "-1", theUnitType, false)
--		trigger.action.outText("dcsCommon - unit name retval " .. aUnit.name, 30)
		dcsCommon.addUnitToGroupData(aUnit, theNewGroup, x, z, heading)
		return theNewGroup
	end
	

	function dcsCommon.arrangeGroupDataIntoFormation(theNewGroup, radius, minDist, formation, innerRadius)
		-- formations:
		--    (default) "line" (left to right along x) -- that is Y direction
		--    "line_v" a line top to bottom -- that is X direction
		--    "chevron" - left to right middle too top
		--    "scattered", "random" -- random, innerRadius used to clear area in center
		-- 	  "circle", "circle_forward" -- circle, forward facing
		--    "circle_in" -- circle, inwarf facing
		--    "circle_out" -- circle, outward facing
		--    "grid", "square", "rect" -- optimal rectangle
		--    "2cols", "2deep" -- 2 columns, n deep 
		--    "2wide" -- 2 columns wide, 2 deep 

		local num = #theNewGroup.units 
		
		-- now do the formation stuff
		-- make sure that they keep minimum  distance 
		if formation == "LINE_V" then 
			-- top to bottom in zone (heding 0). -- will run through x-coordinate 
			-- use entire radius top to bottom 
			local currX = -radius
			local increment = radius * 2/(num - 1) -- MUST NOT TRY WITH 1 UNIT!
			for i=1, num do
			
				local u = theNewGroup.units[i]
				u.x = currX
				currX = currX + increment
			end
		
		elseif formation == "LINE" then 
			-- left to right in zone. runs through Y
			-- left and right are y because at heading 0, forward is x (not y as expected)
			-- if only one, place in middle of circle and be done 
			if num == 1 then 
				-- nothing. just stay in the middle 
			else 
				local currY = -radius
				local increment = radius * 2/(num - 1) -- MUST NOT TRY WITH 1 UNIT!
				for i=1, num do
					local u = theNewGroup.units[i]
					u.y = currY
					currY = currY + increment
				end	
			end 
			
		elseif formation == "CHEVRON" then 
			-- left to right in zone. runs through Y
			-- left and right are y because at heading 0, forward is x (not y as expected)
			local currY = -radius
			local currX = 0
			local incrementY = radius * 2/(num - 1) -- MUST NOT TRY WITH 1 UNIT!
			local incrementX = radius * 2/(num - 1) -- MUST NOT TRY WITH 1 UNIT!
			for i=1, num do
				local u = theNewGroup.units[i]
				u.x = currX
				u.y = currY
				-- calc coords for NEXT iteration
				currY = currY + incrementY -- march left to right
				if i < num / 2 then -- march up
					currX = currX + incrementX 
				elseif i == num / 2 then -- even number, keep height
					currX = currX + 0 
				else 
					currX = currX - incrementX -- march down 
				end 
				-- note: when unit number even, the wedge is sloped. may need an odd/even test for better looks
			end	

		elseif formation == "SCATTERED" or formation == "RANDOM" then 
			-- use randomPointInCircle and tehn iterate over all vehicles for mindelta
			processedUnits = {}
			if not minDist then minDist = 10 end 
			for i=1, num do
				local emergencyBreak = 1 -- prevent endless loop
				local lowDist = 10000
				local uPoint = {}
				local thePoint = {}
				repeat 	-- get random point until mindistance to all is kept or emergencybreak
					thePoint = dcsCommon.randomPointInCircle(radius, innerRadius) -- returns x, 0, z
					-- check if too close to others
					for idx, rUnit in pairs(processedUnits) do -- get min dist to all positioned units
						--trigger.action.outText("rPnt: thePoint =  " .. dcsCommon.point2text(thePoint), 30)
						uPoint.x = rUnit.x
						uPoint.y = 0
						uPoint.z = rUnit.y 
						--trigger.action.outText("rPnt: uPoint =  " .. dcsCommon.point2text(uPoint), 30)
						local dist = dcsCommon.dist(thePoint, uPoint) -- measure distance to unit
						if (dist < lowDist) then lowDist = dist end
					end
					emergencyBreak = emergencyBreak + 1
				until (emergencyBreak > 20) or (lowDist > minDist)
				-- we have random x, y 
				local u = theNewGroup.units[i] -- get unit to position
				u.x = thePoint.x
				u.y = thePoint.z -- z --> y mapping! 
				-- now add the unit to the 'processed' set 
				table.insert(processedUnits, u)
			end	

		elseif dcsCommon.stringStartsWith(formation, "CIRCLE") then
			-- units are arranged on perimeter of circle defined by radius 
			local currAngle = 0
			local angleInc = 2 * 3.14157 / num -- increase per spoke 
			for i=1, num do
				local u = theNewGroup.units[i] -- get unit 
				u.x = radius * math.cos(currAngle)
				u.y = radius * math.sin(currAngle)
				
				-- now baldower out heading 
				-- circle, circle_forward no modifier of heading
				if dcsCommon.stringStartsWith(formation, "CIRCLE_IN") then 
					-- make the heading inward faceing - that's angle + pi
					u.heading = u.heading + currAngle + 3.14157
				elseif dcsCommon.stringStartsWith(formation, "CIRCLE_OUT") then 
					u.heading = u.heading + currAngle + 0
				end

				currAngle = currAngle + angleInc
			end
		elseif formation == "GRID" or formation == "SQUARE" or formation == "RECT" then 
			if num < 2 then return end 
			-- arrange units in an w x h grid
			-- e-g- 12 units = 4 x 3. 
			-- calculate w 
			local w = math.floor(num^(0.5) + 0.5)
			dcsCommon.arrangeGroupInNColumns(theNewGroup, w, radius)
			
		elseif formation == "2DEEP" or formation == "2COLS" then
			if num < 2 then return end 
			-- arrange units in an 2 x h grid
			local w = 2
			dcsCommon.arrangeGroupInNColumnsDeep(theNewGroup, w, radius)

		elseif formation == "2WIDE" then
			if num < 2 then return end 
			-- arrange units in an 2 x h grid
			local w = 2
			dcsCommon.arrangeGroupInNColumns(theNewGroup, w, radius)
		else 
			trigger.action.outText("dcsCommon - unknown formation: " .. formation, 30)
		end
	
	end
	
	function dcsCommon.arrangeGroupInNColumns(theNewGroup, w, radius)
		local num = #theNewGroup.units
		local h = math.floor(num / w)
		if (num % w) > 0 then 
			h = h + 1
		end
		local i = 1
		local xInc = 0 
		if w > 1 then xInc = 2 * radius / (w-1) end
		local yInc = 0
		if h > 1 then yInc = 2 * radius / (h-1) end 
		local currY = radius 
		if h < 2 then currY = 0 end -- special:_ place in Y middle if only one row)
		while h > 0 do 
			local currX = radius 
			local wCnt = w 
			while wCnt > 0 and (i <= num) do 
				local u = theNewGroup.units[i] -- get unit 
				u.x = currX
				u.y = currY
				currX = currX - xInc
				wCnt = wCnt - 1
				i = i + 1
			end
			currY = currY - yInc 
			h = h - 1
		end
	end
	
	function dcsCommon.arrangeGroupInNColumnsDeep(theNewGroup, w, radius)
		local num = #theNewGroup.units
		local h = math.floor(num / w)
		if (num % w) > 0 then 
			h = h + 1
		end
		local i = 1
		local yInc = 0 
		if w > 1 then yInc = 2 * radius / (w-1) end
		local xInc = 0
		if h > 1 then xInc = 2 * radius / (h-1) end 
		local currX = radius 
		if h < 2 then currX = 0 end -- special:_ place in Y middle if only one row)
		while h > 0 do 
			local currY = radius 
			local wCnt = w 
			while wCnt > 0 and (i <= num) do 
				local u = theNewGroup.units[i] -- get unit 
				u.x = currX
				u.y = currY
				currY = currY - yInc
				wCnt = wCnt - 1
				i = i + 1
			end
			currX = currX - xInc 
			h = h - 1
		end
	end
	
	
	function dcsCommon.createGroundGroupWithUnits(name, theUnitTypes, radius, minDist, formation, innerRadius, liveries)
		-- liveries is indexed by typeName and provides alternate livery names 
		-- from default.
		if not minDist then minDist = 4 end -- meters
		if not formation then formation = "line" end 
		if not radius then radius = 30 end -- meters 
		if not innerRadius then innerRadius = 0 end
		if not liveries then liveries = {} end 
		formation = formation:upper()
		-- theUnitTypes can be either a single string or a table of strings
		-- see here for TypeName https://github.com/mrSkortch/DCS-miscScripts/tree/master/ObjectDB
		-- formation defines how the units are going to be arranged in the
		-- formation specified. 
		-- formations:
		--    (default) "line" (left to right along x) -- that is Y direction
		--    "line_V" a line top to bottom -- that is X direction
		--    "chevron" - left to right middle too top
		--    "scattered", "random" -- random, innerRadius used to clear area in center
		-- 	  "circle", "circle_forward" -- circle, forward facing
		--    "circle_in" -- circle, inwarf facing
		--    "circle_out" -- circle, outward facing

		-- first, we create a group
		local theNewGroup = dcsCommon.createEmptyGroundGroupData(name)
		
		-- now add a single unit or multiple units
		if type(theUnitTypes) ~= "table" then 			
			local aUnit = {}
			aUnit = dcsCommon.createGroundUnitData(name .. "-1", theUnitTypes, false)
			dcsCommon.addUnitToGroupData(aUnit, theNewGroup, 0, 0) -- create with data at location (0,0)
			return theNewGroup
		end 

		-- if we get here, theUnitTypes is a table
		-- now loop and create a unit for each table
		local num = 1
		for key, theType in pairs(theUnitTypes) do 
			-- trigger.action.outText("+++dcsC: creating unit " .. name .. "-" .. num .. ": " .. theType, 30)
			local aUnit = dcsCommon.createGroundUnitData(name .. "-"..num, theType, false)
			local theLivery = liveries[theType]
			if theLivery then 
				aUnit.livery_id = theLivery
			end 
			dcsCommon.addUnitToGroupData(aUnit, theNewGroup, 0, 0)
			num = num + 1
		end
		
		dcsCommon.arrangeGroupDataIntoFormation(theNewGroup, radius, minDist, formation, innerRadius)
		return theNewGroup

	
	end

-- create a new group, based on group in mission. Groups coords are 0,0 for group and all
-- x,y and heading
	function dcsCommon.createGroupDataFromLiveGroup(name, newName) 
		if not newName then newName = dcsCommon.uuid("uniqName") end
		-- get access to the group
		local liveGroup = Group.getByName(name)
		if not liveGroup then return nil end
		-- get the categorty
		local cat = liveGroup:getCategory()
		local theNewGroup = {}
		
		-- create a new empty group at (0,0) 
		if cat == Group.Category.AIRPLANE or cat == Group.Category.HELICOPTER then 
			theNewGroup = dcsCommon.createEmptyAircraftGroupData(newName)
		elseif cat == Group.Category.GROUND then
			theNewGroup = dcsCommon.createEmptyGroudGroupData(newName)
		else 
			trigger.action.outText("dcsCommon - unknown category: " .. cat, 30)
			return nil
		end
		

		-- now get all units from live group and create data units
		-- note that unit data for group has x=0, y=0
		liveUnits = liveGroup:getUnits()
		
		for index, theUnit in pairs(liveUnits) do 
			-- for each unit we get the desc 
			local desc = theUnit:getDesc() -- of interest is only typename 
			local newUnit = dcsCommon.createGroundUnitData(dcsCommon.uuid(newName),
														   desc.typeName,
														   false)
			-- we now basically have a ground unit at (0,0) 
			-- add mandatory fields by type
			if cat == Group.Category.AIRPLANE or cat == Group.Category.HELICOPTER then 
				newUnit.alt = 100 -- make it 100m
				newUnit.alt_type = "RADIO" -- AGL
				newUnit.speed = 77 -- m/s --> 150 knots
				newUnit.payload = dcsCommon.createPayload() -- empty payload
				newUnit.callsign = dcsCommon.createCallsign() -- 'enfield11'
				
			elseif cat == Group.Category.GROUND then
				-- we got all we need
			else 

			end			
			
		end
	
	end;
	
	function dcsCommon.pointInDirectionOfPointXYY(dir, dist, p) -- dir in rad, p in XYZ returns XYY 
		local fx = math.cos(dir)
		local fy = math.sin(dir) 
		local p2 = {}
		p2.x = p.x + dist * fx
		p2.y = p.z + dist * fy
		p2.z = p2.y -- make p2 XYY vec2/3 upcast
		return p2
	end

	function dcsCommon.rotatePointAroundOriginRad(inX, inY, angle) -- angle in degrees
		local c = math.cos(angle)
		local s = math.sin(angle)
		local px
		local py 
		px = inX * c - inY * s
		py = inX * s + inY * c
		return px, py		
	end
	
	function dcsCommon.rotatePointAroundOrigin(inX, inY, angle) -- angle in degrees
		local rads =  3.14152 / 180 -- convert to radiants. 
		angle = angle * rads -- turns into rads
		local px, py = dcsCommon.rotatePointAroundOriginRad(inX, inY, angle)
		return px, py		
	end
	
	function dcsCommon.rotatePointAroundPointRad(x, y, px, py, angle)
		x = x - px 
		y = y - py
		x, y = dcsCommon.rotatePointAroundOriginRad(x, y, angle)
		x = x + px 
		y = y + py 
		return x, y
	end

	function dcsCommon.rotatePointAroundPointDeg(x, y, px, py, degrees)
		x, y = dcsCommon.rotatePointAroundPointRad(x, y, px, py, degrees * 3.14152 / 180)
		return x, y
	end

	-- rotates a Vec3-base inPoly on XZ pane around inPoint on XZ pane
 	function dcsCommon.rotatePoly3AroundVec3Rad(inPoly, inPoint, rads)
		local outPoly = {}
		for idx, aVertex in pairs(inPoly) do 
			local x, z = dcsCommon.rotatePointAroundPointRad(aVertex.x, aVertex.z, inPoint.x, inPoint.z, rads)		
			local v3 = {x = x, y = aVertex.y, z = z}
			outPoly[idx] = v3
		end 
		return outPoly 
	end

	function dcsCommon.rotateUnitData(theUnit, degrees, cx, cz)
		if not cx then cx = 0 end
		if not cz then cz = 0 end
		local cy = cz 
		
		local rads = degrees *  3.14152 / 180
		do
			theUnit.x = theUnit.x - cx -- MOVE TO ORIGIN OF ROTATION
			theUnit.y = theUnit.y - cy 				
			theUnit.x, theUnit.y = dcsCommon.rotatePointAroundOrigin(theUnit.x, theUnit.y, degrees)
			theUnit.x = theUnit.x + cx -- MOVE BACK 
			theUnit.y = theUnit.y + cy 				

			-- may also want to increase heading by degrees
			theUnit.heading = theUnit.heading + rads 
		end
	end
	

	function dcsCommon.rotateGroupData(theGroup, degrees, cx, cz)
		if not cx then cx = 0 end
		if not cz then cz = 0 end
		local cy = cz 
		
		local rads = degrees *  3.14152 / 180
		-- turns all units in group around the group's center by degrees.
		-- may also need to turn individual units by same amount
		for i, theUnit in pairs (theGroup.units) do
			theUnit.x = theUnit.x - cx -- MOVE TO ORIGIN OF ROTATION
			theUnit.y = theUnit.y - cy 				
			theUnit.x, theUnit.y = dcsCommon.rotatePointAroundOrigin(theUnit.x, theUnit.y, degrees)
			theUnit.x = theUnit.x + cx -- MOVE BACK 
			theUnit.y = theUnit.y + cy 				

			-- may also want to increase heading by degrees
			theUnit.heading = theUnit.heading + rads 
			if theUnit.psi then 
				theUnit.psi = -theUnit.heading 
			end
		end
	end

	function dcsCommon.offsetGroupData(theGroup, dx, dy)
		-- add dx and dy to group's and all unit's coords
		for i, theUnit in pairs (theGroup.units) do 
			theUnit.x = theUnit.x + dx
			theUnit.y = theUnit.y + dy
		end
		
		theGroup.x = theGroup.x + dx
		theGroup.y = theGroup.y + dy 
	end
	
	function dcsCommon.moveGroupDataTo(theGroup, xAbs, yAbs)
		local dx = xAbs-theGroup.x
		local dy = yAbs-theGroup.y
		dcsCommon.offsetGroupData(theGroup, dx, dy)
	end
	
	-- static objectr shapes and types are defined here
	-- https://github.com/mrSkortch/DCS-miscScripts/tree/master/ObjectDB/Statics
	
	function dcsCommon.createStaticObjectData(name, objType, heading, dead, cargo, mass)
		local staticObj = {}
		if not heading then heading = 0 end 
		if not dead then dead = false end 
		if not cargo then cargo = false end 
		objType = dcsCommon.trim(objType) 
		
		staticObj.heading = heading
		-- staticObj.groupId = 0
		-- staticObj.shape_name = shape -- e.g. H-Windsock_RW
		staticObj.type = objType  -- e.g. Windsock
		-- ["unitId"] = 3,
		staticObj.rate = 1 -- score when killed
		staticObj.name = name
		-- staticObj.category = "Fortifications",
		staticObj.y = 0
		staticObj.x = 0
		staticObj.dead = dead
		staticObj.canCargo = cargo -- to cargo
		if cargo then 
			if not mass then mass = 1234 end 
			staticObj.mass = mass -- to cargo
		end
		return staticObj
	end
	
	function dcsCommon.createStaticObjectDataAt(loc, name, objType, heading, dead)
		local theData = dcsCommon.createStaticObjectData(name, objType, heading, dead)
		theData.x = loc.x
		theData.y = loc.z 
		return theData
	end
	
	function dcsCommon.createStaticObjectForCoalitionAtLocation(theCoalition, loc, name, objType, heading, dead) 
		if not heading then heading = math.random(360) * 3.1415 / 180 end
		local theData = dcsCommon.createStaticObjectDataAt(loc, name, objType, heading, dead)
		local theStatic = coalition.addStaticObject(theCoalition, theData)
		return theStatic
	end
	
	function dcsCommon.createStaticObjectForCoalitionInRandomRing(theCoalition, objType, x, z, innerRadius, outerRadius, heading, alive) 
		if not outerRadius then outerRadius = innerRadius end
		if not heading then heading = math.random(360) * 3.1415 / 180 end
		local dead = not alive
		local p = dcsCommon.randomPointInCircle(outerRadius, innerRadius, x, z)
		local theData = dcsCommon.createStaticObjectData(dcsCommon.uuid("static"), objType, heading, dead)
		theData.x = p.x
		theData.y = p.z 
		
		local theStatic = coalition.addStaticObject(theCoalition, theData)
		return theStatic
	end
	
	
	
	function dcsCommon.linkStaticDataToUnit(theStatic, theUnit, dx, dy, heading)
		if not theStatic then 
			trigger.action.OutText("+++dcsC: NIL theStatic on linkStatic!", 30)
			return 
		end
		-- NOTE: we may get current heading and subtract/add 
		-- to original heading 
		local rotX, rotY = dcsCommon.rotatePointAroundOrigin(dx, dy, -heading)
		
		if not theUnit then return end
		if not theUnit:isExist() then return end 
		theStatic.linkOffset = true 
		theStatic.linkUnit = theUnit:getID()
		local unitPos = theUnit:getPoint()
		local offsets = {}
		offsets.x = rotX  
		offsets.y = rotY 
		offsets.angle = 0
		theStatic.offsets = offsets
	end
	
	function dcsCommon.offsetStaticData(theStatic, dx, dy)
		theStatic.x = theStatic.x + dx
		theStatic.y = theStatic.y + dy
		-- now check if thre is a route (for linked objects)
		if theStatic.route then 
			-- access points[1] x and y and copy from main
			theStatic.route.points[1].x = theStatic.x
			theStatic.route.points[1].y = theStatic.y
		end
	end
	
	function dcsCommon.moveStaticDataTo(theStatic, x, y)
		theStatic.x = x
		theStatic.y = y
		-- now check if thre is a route (for linked objects)
		if theStatic.route then 
			-- access points[1] x and y and copy from main
			theStatic.route.points[1].x = theStatic.x
			theStatic.route.points[1].y = theStatic.y
		end

	end

function dcsCommon.synchGroupData(inGroupData) -- update group data block by 
-- comparing it to spawned group and update units by x, y, heding and isExist 
-- modifies inGroupData!
	if not inGroupData then return end 
	-- groupdata from game, NOT MX DATA!
	-- we synch the units and their coords 
	local livingUnits = {}
	for idx, unitData in pairs(inGroupData.units) do 
		local theUnit = Unit.getByName(unitData.name)
		if theUnit and theUnit:isExist() and theUnit:getLife()>1 then 
			-- update x and y and heading
			local pos = theUnit:getPoint()
			unitData.unitId = theUnit:getID()
			unitData.x = pos.x 
			unitData.y = pos.z -- !!!!
			unitData.heading = dcsCommon.getUnitHeading(gUnit)
			table.insert(livingUnits, unitData)
		end
	end
	inGroupData.units = livingUnits 
end

--
--
-- M I S C   M E T H O D S 
--
--

-- as arrayContainsString, except it includes wildcard matches if EITHER 
-- ends on "*"
	function dcsCommon.wildArrayContainsString(theArray, theString, caseSensitive) 
		if not theArray then return false end
		if not theString then return false end
		if not caseSensitive then caseSensitive = false end 
		if type(theArray) ~= "table" then 
			trigger.action.outText("***arrayContainsString: theArray is not type table but <" .. type(theArray) .. ">", 30)
		end
		if not caseSensitive then theString = string.upper(theString) end 
		
		local wildIn = dcsCommon.stringEndsWith(theString, "*")
		if wildIn then dcsCommon.removeEnding(theString, "*") end 
		for idx, theElement in pairs(theArray) do -- i = 1, #theArray do 
			if not caseSensitive then theElement = string.upper(theElement) end 
			local wildEle = dcsCommon.stringEndsWith(theElement, "*")
			if wildEle then theElement = dcsCommon.removeEnding(theElement, "*") end 

			if wildEle and wildIn then 
				-- both end on wildcards, partial match for both
				if dcsCommon.stringStartsWith(theElement, theString) then return true end 
				if dcsCommon.stringStartsWith(theString, theElement) then return true end 
			elseif wildEle then 
				-- Element is a wildcard, partial match 
				if dcsCommon.stringStartsWith(theString, theElement) then return true end

			elseif wildIn then
				-- theString is a wildcard. partial match 
				if dcsCommon.stringStartsWith(theElement, theString) then return true end
			else
				-- standard: no wildcards, full match
				if theElement == theString then return true end 
			end
			
		end
		return false 
	end


	function dcsCommon.arrayContainsString(theArray, theString) 
		if not theArray then return false end
		if not theString then return false end
		if type(theArray) ~= "table" then 
			trigger.action.outText("***arrayContainsString: theArray is not type table but <" .. type(theArray) .. ">", 30)
		end
		for i = 1, #theArray do 
			if theArray[i] == theString then return true end 
		end
		return false 
	end
	
	function dcsCommon.splitString(inputstr, sep) 
        if sep == nil then
            sep = "%s"
        end
		if inputstr == nil then 
			inputstr = ""
		end
		
        local t={}
        for str in string.gmatch(inputstr, "([^"..sep.."]+)") do
			table.insert(t, str)
        end
        return t
	
	end
	
	function dcsCommon.trimFront(inputstr) 
		if not inputstr then return nil end 
		local s = inputstr
		while string.len(s) > 1 and string.sub(s, 1, 1) == " " do 
			local snew = string.sub(s, 2) -- all except first
			s = snew
		end
		return s
	end
	
	function dcsCommon.trimBack(inputstr)
		if not inputstr then return nil end 
		local s = inputstr
		while string.len(s) > 1 and string.sub(s, -1) == " " do 
			local snew = string.sub(s, 1, -2) -- all except last
			s = snew
		end
		return s
	end
	
	function dcsCommon.trim(inputstr) 
		local t1 = dcsCommon.trimFront(inputstr)
		local t2 = dcsCommon.trimBack(t1)
		return t2
	end
	
	function dcsCommon.trimArray(theArray)
		local trimmedArray = {}
		for idx, element in pairs(theArray) do 
			local tel = dcsCommon.trim(element)
			table.insert(trimmedArray, tel)
		end
		return trimmedArray
	end
	
	function dcsCommon.string2Array(inString, deli, uCase)
		if not inString then return {} end 
		if not deli then return {} end 
		if not uCase then uCase = false end
		if uCase then inString = string.upper(inString) end
		inString = dcsCommon.trim(inString)
		if dcsCommon.containsString(inString, deli) then 
			local a = dcsCommon.splitString(inString, deli)
			a = dcsCommon.trimArray(a)
			return a 
		else 
			return {inString}
		end
	end
	
	function dcsCommon.array2string(inArray, deli)
		if not deli then deli = ", " end
		if type(inArray) ~= "table" then return "<err in array2string: not an array>" end
		local s = ""
		local count = 0
		for idx, ele in pairs(inArray) do
			if count > 0 then s = s .. deli .. " " end
			s = s .. ele
			count = count + 1
		end
		return s
	end
	
	function dcsCommon.stripLF(theString)
		return theString:gsub("[\r\n]", "")
	end
	
	function dcsCommon.removeBlanks(theString)
		return theString:gsub("%s", "")
	end
	
	function dcsCommon.stringIsPositiveNumber(theString)
		-- only full integer positive numbers supported 
		if not theString then return false end 
--		if theString == "" then return false end 
		for i = 1, #theString do 
			local c = theString:sub(i,i)
			if c < "0" or c > "9" then return false end 
		end
		return true 
	end
	
	function dcsCommon.stringStartsWithDigit(theString)
		if #theString < 1 then return false end 
		local c = string.sub(theString, 1, 1) 
		return c >= "0" and c <= "9" 
	end
	
	function dcsCommon.stringStartsWithLetter(theString)
		if #theString < 1 then return false end 
		local c = string.sub(theString, 1, 1)
		if c >= "a" and c <= "z" then return true end  
		if c >= "A" and c <= "Z" then return true end 
		return false 
	end
	
	function dcsCommon.stringStartsWith(theString, thePrefix, caseInsensitive)
		if not theString then return false end 
		if not thePrefix then return false end 
		if not caseInsensitive then caseInsensitive = false end 
		
		if caseInsensitive then 
			theString = string.upper(theString)
			thePrefix = string.upper(thePrefix)
		end
		-- superseded: string.find (s, pattern [, init [, plain]]) solves the problem  
		local i, j = string.find(theString, thePrefix, 1, true)
		return (i == 1)
	end
	
	function dcsCommon.removePrefix(theString, thePrefix)
		if not dcsCommon.stringStartsWith(theString, thePrefix) then 
			return theString
		end;
		return theString:sub(1 + #thePrefix)
	end
	
	function dcsCommon.stringEndsWith(theString, theEnding)
		return theEnding == "" or theString:sub(-#theEnding) == theEnding
	end
	
	function dcsCommon.removeEnding(theString, theEnding) 
		if not dcsCommon.stringEndsWith(theString, theEnding) then 
			return theString
		end
		return theString:sub(1, #theString - #theEnding)
	end
	
	function dcsCommon.containsString(inString, what, caseSensitive)
		if (not caseSensitive) then 
			inString = string.upper(inString)
			what = string.upper(what)
		end
		if inString == what then return true end -- when entire match 
		return string.find(inString, what, 1, true) -- 1, true means start at 1, plaintext
	end
	
	function dcsCommon.bool2Text(theBool) 
		if not theBool then theBool = false end 
		if theBool then return "true" end 
		return "false"
	end
	
	function dcsCommon.bool2YesNo(theBool)
		if not theBool then 
			theBool = false
			return "NIL"
		end 
		if theBool then return "yes" end 
		return "no"
	end
	
	function dcsCommon.bool2Num(theBool)
		if not theBool then theBool = false end 
		if theBool then return 1 end 
		return 0
	end

	function dcsCommon.point2text(p, intsOnly) 
		if not intsOnly then intsOnly = false end 
		if not p then return "<!NIL!>" end 
		local t = "[x="
		if intsOnly then 
			if p.x then t = t .. math.floor(p.x) .. ", " else t = t .. "<nil>, " end 
			if p.y then t = t .. "y=" .. math.floor(p.y) .. ", " else t = t .. "y=<nil>, " end 
			if p.z then t = t .. "z=" .. math.floor(p.z) .. "]" else t = t .. "z=<nil>]" end 
		else 
			if p.x then t = t .. p.x .. ", " else t = t .. "<nil>, " end 
			if p.y then t = t .. "y=" .. p.y .. ", " else t = t .. "y=<nil>, " end 
			if p.z then t = t .. "z=" .. p.z .. "]" else t = t .. "z=<nil>]" end 
		end
		return t 
	end

	function dcsCommon.string2GroupCat(inString)

		if not inString then return 2 end -- default ground 
		inString = inString:lower()
		inString = dcsCommon.trim(inString)

		local catNum = tonumber(inString)
		if catNum then 
			if catNum < 0 then catNum = 0 end 
			if catNum > 4 then catNum = 4 end 
			return catNum 
		end
	
		catNum = 2 -- ground default 
		if dcsCommon.stringStartsWith(inString, "grou") then catNum = 2 end 
		if dcsCommon.stringStartsWith(inString, "air") then catNum = 0 end
		if dcsCommon.stringStartsWith(inString, "hel") then catNum = 1 end
		if dcsCommon.stringStartsWith(inString, "shi") then catNum = 3 end
		if dcsCommon.stringStartsWith(inString, "trai") then catNum = 4 end

		return catNum
	end

	function dcsCommon.string2ObjectCat(inString)

		if not inString then return 3 end -- default static 
		inString = inString:lower()
		inString = dcsCommon.trim(inString)

		local catNum = tonumber(inString)
		if catNum then 
			if catNum < 0 then catNum = 0 end 
			if catNum > 6 then catNum = 6 end 
			return catNum 
		end
	
		catNum = 3 -- static default 
		if dcsCommon.stringStartsWith(inString, "uni") then catNum = 1 end 
		if dcsCommon.stringStartsWith(inString, "wea") then catNum = 2 end
		if dcsCommon.stringStartsWith(inString, "bas") then catNum = 4 end
		if dcsCommon.stringStartsWith(inString, "sce") then catNum = 5 end
		if dcsCommon.stringStartsWith(inString, "car") then catNum = 6 end

		return catNum
	end

	function dcsCommon.menu2text(inMenu)
		if not inMenu then return "<nil>" end
		local s = ""
		for n, v in pairs(inMenu) do 
			if type(v) == "string" then 
				if s == "" then s = "[" .. v .. "]"  else 
					s = s .. " | [" .. type(v) .. "]" end
			else 
				if s == "" then s = "[<" .. type(v) .. ">]"  else
					s = s .. " | [<" .. type(v) .. ">]" end
			end
		end
		return s
	end

	-- recursively show the contents of a variable
	function dcsCommon.dumpVar(key, value, prefix, inrecursion)
		if not inrecursion then 
			-- output a marker to find in the log / screen
			env.info("*** dcsCommon vardump START")
		end
		if not value then value = "nil" end
		if not prefix then prefix = "" end
		prefix = " " .. prefix
		if type(value) == "table" then 
			env.info(prefix .. key .. ": [ ")
			-- iterate through all kvp
			for k,v in pairs (value) do
				dcsCommon.dumpVar(k, v, prefix, true)
			end
			env.info(prefix .. " ] - end " .. key)
			
		elseif type(value) == "boolean" then 
			local b = "false"
			if value then b = "true" end
			env.info(prefix .. key .. ": " .. b)
			
		else -- simple var, show contents, ends recursion
			env.info(prefix .. key .. ": " .. value)
		end
		
		if not inrecursion then 
			-- output a marker to find in the log / screen
			trigger.action.outText("=== dcsCommon vardump end", 30)
			env.info("=== dcsCommon vardump end")
		end
	end
	
	function dcsCommon.dumpVar2Str(key, value, prefix, inrecursion)
		-- dumps to screen, not string 
		if not inrecursion then 
			-- output a marker to find in the log / screen
			trigger.action.outText("*** dcsCommon vardump START",30)
		end
		if not value then value = "nil" end
		if not prefix then prefix = "" end
		prefix = " " .. prefix
		if getmetatable(value) then 
			if type(value) == "string" then 
			else 
				trigger.action.outText(prefix .. key (" .. type(value) .. ") .. " HAS META", 30)
			end
		end
		if type(value) == "table" then 
			trigger.action.outText(prefix .. key .. ": [ ", 30)
			-- iterate through all kvp
			for k,v in pairs (value) do
				dcsCommon.dumpVar2Str(k, v, prefix, true)
			end
			trigger.action.outText(prefix .. " ] - end " .. key, 30)
			
		elseif type(value) == "boolean" then 
			local b = "false"
			if value then b = "true" end
			trigger.action.outText(prefix .. key .. ": " .. b, 30)
			
		else -- simple var, show contents, ends recursion
			trigger.action.outText(prefix .. key .. ": " .. value, 30)
		end
		
		if not inrecursion then 
			-- output a marker to find in the log / screen
			trigger.action.outText("=== dcsCommon vardump end", 30)
		end
	end
		
	function dcsCommon.numberUUID()
		dcsCommon.simpleUUID = dcsCommon.simpleUUID + 1
		return dcsCommon.simpleUUID
	end

	function dcsCommon.uuid(prefix)
		--dcsCommon.uuIdent = dcsCommon.uuIdent + 1
		if not prefix then prefix = dcsCommon.uuidStr end
		return prefix .. "-" .. dcsCommon.numberUUID() -- dcsCommon.uuIdent
	end
	
	function dcsCommon.event2text(id) 
		if not id then return "error" end
		if id == 0 then return "invalid" end
		-- translate the event id to text
		local events = {"shot", "hit", "takeoff", "land",
						"crash", "eject", "refuel", "dead", -- 8
						"pilot dead", "base captured", "mission start", "mission end", -- 12
						"took control", "refuel stop", "birth", "human failure", -- 16 
						"det. failure", "engine start", "engine stop", "player enter unit", -- 20
						"player leave unit", "player comment", "start shoot", "end shoot", -- 24
						"mark add", "mark changed", "mark removed", "kill", -- 28 
						"score", "unit lost", "land after eject", "Paratrooper land", -- 32 
						"chair discard after eject", "weapon add", "trigger zone", "landing quality mark", -- 36
						"BDA", "AI Abort Mission", "DayNight", "Flight Time", -- 40
						"Pilot Suicide", "player cap airfield", "emergency landing", "unit create task", -- 44
						"unit delete task", "Simulation start", "weapon rearm", "weapon drop", -- 48
						"unit task timeout", "unit task stage", -- 50
						"subtask score", "extra score", "mission restart", "winner", 
						"postponed takeoff", "postponed land", -- 56
						"max"}
		if id > #events then return "Unknown (ID=" .. id .. ")" end
		return events[id]
	end

	function dcsCommon.smokeColor2Text(smokeColor)
		if (smokeColor == 0) then return "Green" end
		if (smokeColor == 1) then return "Red" end
		if (smokeColor == 2) then return "White" end
		if (smokeColor == 3) then return "Orange" end
		if (smokeColor == 4) then return "Blue" end
		
		return ("unknown: " .. smokeColor)
	end
	
	function dcsCommon.flareColor2Text(flareColor)
		if (flareColor == 0) then return "Green" end
		if (flareColor == 1) then return "Red" end
		if (flareColor == 2) then return "White" end
		if (flareColor == 3) then return "Yellow" end
		if (flareColor < 0) then return "Random" end 
		return ("unknown: " .. flareColor)
	end
	
	function dcsCommon.smokeColor2Num(smokeColor)
		if not smokeColor then smokeColor = "green" end 
		if type(smokeColor) ~= "string" then return 0 end 
		smokeColor = smokeColor:lower()
		if (smokeColor == "green") then return 0 end 
		if (smokeColor == "red") then return 1 end 
		if (smokeColor == "white") then return 2 end 
		if (smokeColor == "orange") then return 3 end 
		if (smokeColor == "blue") then return 4 end 
		return 0
	end

	function dcsCommon.flareColor2Num(flareColor)
		if not flareColor then flareColor = "green" end 
		if type(flareColor) ~= "string" then return 0 end 
		flareColor = flareColor:lower()
		if (flareColor == "green") then return 0 end 
		if (flareColor == "red") then return 1 end 
		if (flareColor == "white") then return 2 end 
		if (flareColor == "yellow") then return 3 end 
		if (flareColor == "random") then return -1 end 
		if (flareColor == "rnd") then return -1 end 
		return 0
	end

	
	function dcsCommon.markPointWithSmoke(p, smokeColor)
		if not smokeColor then smokeColor = 0 end 
		local x = p.x 
		local z = p.z -- do NOT change the point directly
		-- height-correct
		local y = land.getHeight({x = x, y = z})
		local newPoint= {x = x, y = y + 2, z = z}
		trigger.action.smoke(newPoint, smokeColor)
	end

-- based on buzzer1977's idea, channel is number, eg in 74X, channel is 74, mode is "X"
	function dcsCommon.tacan2freq(channel, mode)	
		if not mode then mode = "X" end 
		if not channel then channel = 1 end 
		if type(mode) ~= "string" then mode = "X" end 
		mode = mode:upper()
		local offset = 1000000 * channel
		if channel < 64 then 
			if mode == "Y" then
				return 1087000000 + offset
			end
			return 961000000 + offset -- mode x
		end
	
		if mode == "Y" then
			return 961000000 + offset
		end
		return 1087000000 + offset -- mode x
	end
	
	function dcsCommon.processHMS(msg, delta)
		local rS = math.floor(delta)
		local remainS = tostring(rS)
		local rM = math.floor(delta/60)
		local remainM = tostring(rM)
		local rH = math.floor(delta/3600)
		local remainH = tostring(rH)
		local hmsH = remainH 
		if rH < 10 then hmsH = "0" .. hmsH end 
		
		local hmsCount = delta - (rH * 3600) -- mins left 
		local mins = math.floor (hmsCount / 60)
		local hmsM = tostring(mins)
		if mins < 10 then hmsM = "0" .. hmsM end 
		
		hmsCount = hmsCount - (mins * 60) 
		local secs = math.floor(hmsCount)
		local hmsS = tostring(secs)
		if secs < 10 then hmsS = "0" .. hmsS end 
		
		msg = string.gsub(msg, "<s>", remainS)
		msg = string.gsub(msg, "<m>", remainM)
		msg = string.gsub(msg, "<h>", remainH)
		
		msg = string.gsub(msg, "<:s>", hmsS)
		msg = string.gsub(msg, "<:m>", hmsM)
		msg = string.gsub(msg, "<:h>", hmsH)
		
		return msg 
	end
	
	function dcsCommon.nowString()
		local absSecs = timer.getAbsTime()-- + env.mission.start_time
		while absSecs > 86400 do 
			absSecs = absSecs - 86400 -- subtract out all days 
		end
		return dcsCommon.processHMS("<:h>:<:m>:<:s>", absSecs)
	end
	
	function dcsCommon.str2num(inVal, default) 
		if not default then default = 0 end
		if not inVal then return default end
		if type(inVal) == "number" then return inVal end 				
		local num = nil
		if type(inVal) == "string" then num = tonumber(inVal) end
		if not num then return default end
		return num
	end
	
	function dcsCommon.stringRemainsStartingWith(theString, startingWith)
		-- find the first position where startingWith starts 
		local pos = theString:find(startingWith)
		if not pos then return theString end 
		-- now return the entire remainder of the string from pos 
		local nums = theString:len() - pos + 1
		return theString:sub(-nums)
	end

--
--
-- V E C T O R   M A T H 
--
--

function dcsCommon.vAdd(a, b) 
	local r = {}
	if not a then a = {x = 0, y = 0, z = 0} end
	if not b then b = {x = 0, y = 0, z = 0} end
	r.x = a.x + b.x 
	r.y = a.y + b.y 
	if a.z and b.z then 
		r.z = a.z + b.z 
	end 
	return r 
end

function dcsCommon.vSub(a, b) 
	local r = {}
	if not a then a = {x = 0, y = 0, z = 0} end
	if not b then b = {x = 0, y = 0, z = 0} end
	r.x = a.x - b.x 
	r.y = a.y - b.y 
	if a.z and b.z then 
		r.z = a.z - b.z 
	end 
	return r 
end

function dcsCommon.vMultScalar(a, f) 
	local r = {}
	if not a then a = {x = 0, y = 0, z = 0} end
	if not f then f = 0 end
	r.x = a.x * f 
	r.y = a.y * f 
	if a.z then 
		r.z = a.z * f
    end		
	return r 
end

function dcsCommon.vLerp (a, b, t)
	if not a then a = {x = 0, y = 0, z = 0} end
	if not b then b = {x = 0, y = 0, z = 0} end
	
	local d = dcsCommon.vSub(b, a)
	local dt = dcsCommon.vMultScalar(d, t)
	local r = dcsCommon.vAdd(a, dt)
	return r
end

function dcsCommon.mag(x, y, z) 
	if not x then x = 0 end
	if not y then y = 0 end 
	if not z then z = 0 end 
	
	return (x * x + y * y + z * z)^0.5
end

function dcsCommon.vMag(a) 
	if not a then return 0 end 
	if not a.x then a.x = 0 end 
	if not a.y then a.y = 0 end 
	if not a.z then a.z = 0 end
	return dcsCommon.mag(a.x, a.y, a.z) 
end

function dcsCommon.magSquare(x, y, z) 
	if not x then x = 0 end
	if not y then y = 0 end 
	if not z then z = 0 end 
	
	return (x * x + y * y + z * z)
end

function dcsCommon.vNorm(a) 
	if not a then return {x = 0, y = 0, z = 0} end 
	m = dcsCommon.vMag(a)
	if m <= 0 then return {x = 0, y = 0, z = 0} end 
	local r = {}
	r.x = a.x / m 
	r.y = a.y / m 
	r.z = a.z / m
	return r 
end

function dcsCommon.dot (a, b) 
	if not a then a = {} end 
	if not a.x then a.x = 0 end 
	if not a.y then a.y = 0 end 
	if not a.z then a.z = 0 end
	if not b then b = {} end 
	if not b.x then b.x = 0 end 
	if not b.y then b.y = 0 end 
	if not b.z then b.z = 0 end 
	
	return a.x * b.x + a.y * b.y + a.z * b.z 
end
--
-- UNIT MISC
-- 
function dcsCommon.isSceneryObject(theUnit)
	if not theUnit then return false end
	return Object.getCategory(theUnit) == 5 
--	return theUnit.getCoalition == nil -- scenery objects do not return a coalition 
end

function dcsCommon.isTroopCarrierType(theType, carriers)
	if not theType then return false end 
	if not carriers then carriers = dcsCommon.troopCarriers 
	end 
	-- remember that arrayContainsString is case INsensitive by default 
	if dcsCommon.wildArrayContainsString(carriers, theType) then 
		-- may add additional tests before returning true
		return true
	end
	
	-- see if user wanted 'any' or 'all' supported
	if dcsCommon.arrayContainsString(carriers, "any") then 
		return true 
	end 
	
	if dcsCommon.arrayContainsString(carriers, "all") then 
		return true 
	end 
	
	return false
end

function dcsCommon.isTroopCarrier(theUnit, carriers)
	-- return true if conf can carry troups
	if not theUnit then return false end 
	local uType = theUnit:getTypeName()
	return dcsCommon.isTroopCarrierType(uType, carriers) 
end


function dcsCommon.getAllExistingPlayerUnitsRaw()
	local apu = {}
	for idx, theSide in pairs(dcsCommon.coalitionSides) do
		local thePlayers = coalition.getPlayers(theSide) 
		for idy, theUnit in pairs (thePlayers) do 
			if theUnit and theUnit:isExist() then 
				table.insert(apu, theUnit)
			end
		end
	end
	return apu 
end

function dcsCommon.getAllExistingPlayersAndUnits() -- units indexed by player name
-- designed to replace cases for cfxPlayer.getAllPlayer invocations
	local apu = {}
	for idx, theSide in pairs(dcsCommon.coalitionSides) do
		local thePlayers = coalition.getPlayers(theSide) 
		for idy, theUnit in pairs (thePlayers) do 
			if theUnit and theUnit:isExist() then 
				local pName = theUnit:getPlayerName()
				apu[pName] = theUnit
			end
		end
	end
	return apu 
end

function dcsCommon.getUnitAlt(theUnit)
	if not theUnit then return 0 end
	if not Unit.isExist(theUnit) then return 0 end -- safer 
	local p = theUnit:getPoint()
	return p.y 
end

function dcsCommon.getUnitAGL(theUnit)
	if not theUnit then return 0 end
	if not Unit.isExist(theUnit) then return 0 end -- safe fix
	local p = theUnit:getPoint()
	local alt = p.y 
	local loc = {x = p.x, y = p.z}
	local landElev = land.getHeight(loc)
	return alt - landElev
end 

function dcsCommon.getUnitSpeed(theUnit)
	if not theUnit then return 0 end
	if not Unit.isExist(theUnit) then return 0 end 
	local v = theUnit:getVelocity()
	return dcsCommon.mag(v.x, v.y, v.z)
end

-- closing velocity of u1 and u2, seen from u1
function dcsCommon.getClosingVelocity(u1, u2)
	if not u1 then return 0 end 
	if not u2 then return 0 end 
	if not u1:isExist() then return 0 end 
	if not u2:isExist() then return 0 end 
	local v1 = u1:getVelocity()
	local v2 = u2:getVelocity()
	local dV = dcsCommon.vSub(v1,v2)
	local a = u1:getPoint()
	local b = u2:getPoint() 
	local aMinusB = dcsCommon.vSub(a,b) -- vector from u2 to u1
	local abMag = dcsCommon.vMag(aMinusB) -- distance u1 to u2 
	if abMag < .0001 then return 0 end 
	-- project deltaV onto vector from u2 to u1 
	local vClose = dcsCommon.dot(dV, aMinusB) / abMag 
	return vClose 
end

function dcsCommon.getGroupAvgSpeed(theGroup)
	if not theGroup then return 0 end 
	if not dcsCommon.isGroupAlive(theGroup) then return 0 end 
	local totalSpeed = 0
	local cnt = 0 
	local livingUnits = theGroup:getUnits()
	for idx, theUnit in pairs(livingUnits) do 
		cnt = cnt + 1
		totalSpeed = totalSpeed + dcsCommon.getUnitSpeed(theUnit)
	end 
	if cnt == 0 then return 0 end 
	return totalSpeed / cnt 
end
 
function dcsCommon.getGroupMaxSpeed(theGroup)
	if not theGroup then return 0 end 
	if not dcsCommon.isGroupAlive(theGroup) then return 0 end 
	local maxSpeed = 0
	local livingUnits = theGroup:getUnits()
	for idx, theUnit in pairs(livingUnits) do 
		currSpeed = dcsCommon.getUnitSpeed(theUnit)
		if currSpeed > maxSpeed then maxSpeed = currSpeed end 
	end 
	return maxSpeed
end 

function dcsCommon.getUnitHeading(theUnit)
	if not theUnit then return 0 end 
	if not theUnit:isExist() then return 0 end 
	local pos = theUnit:getPosition() -- returns three vectors, p is location

	local heading = math.atan2(pos.x.z, pos.x.x)
	-- make sure positive only, add 360 degrees
	if heading < 0 then
		heading = heading + 2 * math.pi	-- put heading in range of 0 to 2*pi
	end
	return heading 
end

function dcsCommon.getUnitHeadingDegrees(theUnit)
	local heading = dcsCommon.getUnitHeading(theUnit)
	return heading * 57.2958 -- 180 / math.pi 
end

function dcsCommon.typeIsInfantry(theType)
	local isInfantry =  
				dcsCommon.containsString(theType, "infantry", false) or 
				dcsCommon.containsString(theType, "paratrooper", false) or
				dcsCommon.containsString(theType, "stinger", false) or
				dcsCommon.containsString(theType, "manpad", false) or
				dcsCommon.containsString(theType, "soldier", false) or 
				dcsCommon.containsString(theType, "SA-18 Igla", false)
	return isInfantry
end

function dcsCommon.unitIsInfantry(theUnit)
	if not theUnit then return false end 
	if not theUnit:isExist() then return end
	local theType = theUnit:getTypeName()
	return dcsCommon.typeIsInfantry(theType)
end

function dcsCommon.coalition2county(inCoalition)
	-- simply return UN troops for 0 neutral,
	-- joint red for 1  red
	-- joint blue for 2 blue 
	if inCoalition == 1 then return 81 end -- cjtf red
	if inCoalition == 2 then return 80 end -- blue 
	if type(inCoalition) == "string" then 
			inCoalition = inCoalition:lower()
			if inCoalition == "red" then return 81 end
			if inCoalition == "blue" then return 80 end
	end
		
	trigger.action.outText("+++dcsC: coalition2county in (" .. inCoalition .. ") converts to UN (82)!", 30)
	return 82 -- UN 
	
end

function dcsCommon.coalition2Text(coa)
	if not coa then return "!nil!" end 
	if coa == 0 then return "NEUTRAL" end 
	if coa == 1 then return "RED" end 
	if coa == 2 then return "BLUE" end 
	return "?UNKNOWN?"
end

function dcsCommon.latLon2Text(lat, lon)
	-- inspired by mist, thanks Grimes!
	-- returns two strings: lat and lon 
	
	-- determine hemispheres by sign
	local latHemi, lonHemi
	if lat > 0 then latHemi = 'N' else latHemi = 'S' end
	if lon > 0 then lonHemi = 'E' else lonHemi = 'W' end

	-- remove sign since we have hemi
	lat = math.abs(lat)
	lon = math.abs(lon)

	-- calc deg / mins 
	local latDeg = math.floor(lat)
	local latMin = (lat - latDeg) * 60
	local lonDeg = math.floor(lon)
	local lonMin = (lon - lonDeg) * 60

	-- calc seconds 
	local rawLatMin = latMin
	latMin = math.floor(latMin)
	local latSec = (rawLatMin - latMin) * 60
	local rawLonMin = lonMin
	lonMin = math.floor(lonMin)
	local lonSec = (rawLonMin - lonMin) * 60

	-- correct for rounding errors 
	if latSec >= 60 then
		latSec = latSec - 60
		latMin = latMin + 1
	end
	if lonSec >= 60 then
		lonSec = lonSec - 60
		lonMin = lonMin + 1
	end

	-- prepare string output 
	local secFrmtStr = '%06.3f'
	local lat = string.format('%02d', latDeg) .. '°' .. string.format('%02d', latMin) .. "'" .. string.format(secFrmtStr, latSec) .. '"' .. latHemi
	local lon = string.format('%02d', lonDeg) .. '°' .. string.format('%02d', lonMin) .. "'" .. string.format(secFrmtStr, lonSec) .. '"' .. lonHemi
	return lat, lon  
end

-- get mission name. If mission file name without ".miz"
function dcsCommon.getMissionName()
	local mn = net.dostring_in("gui", "return DCS.getMissionName()")
	return mn
end

function dcsCommon.numberArrayFromString(inString, default) -- moved from cfxZones
	if not default then default = 0 end 
	if string.len(inString) < 1 then 
		trigger.action.outText("+++dcsCommon: empty numbers", 30)
		return {default, } 
	end
	
	local flags = {}
	local rawElements = dcsCommon.splitString(inString, ",")
	-- go over all elements 
	for idx, anElement in pairs(rawElements) do 
		anElement = dcsCommon.trim(anElement)
		if dcsCommon.stringStartsWithDigit(anElement) and dcsCommon.containsString(anElement, "-") then 
			-- interpret this as a range
			local theRange = dcsCommon.splitString(anElement, "-")
			local lowerBound = theRange[1]
			lowerBound = tonumber(lowerBound)
			local upperBound = theRange[2]
			upperBound = tonumber(upperBound)
			if lowerBound and upperBound then
				-- swap if wrong order
				if lowerBound > upperBound then 
					local temp = upperBound
					upperBound = lowerBound
					lowerBound = temp 
				end
				-- now add add numbers to flags
				for f=lowerBound, upperBound do 
					table.insert(flags, tostring(f))
				end
			else
				-- bounds illegal
				trigger.action.outText("+++dcsCommon: ignored range <" .. anElement .. "> (range)", 30)
			end
		else
			-- single number
			f = dcsCommon.trim(anElement)
			f = tonumber(f)
			if f then 
				table.insert(flags, f)
			end
		end
	end
	if #flags < 1 then flags = {default, } end 
	return flags
end 

function dcsCommon.flagArrayFromString(inString, verbose)
	if not verbose then verbose = false end 
	
	if verbose then 
		trigger.action.outText("+++flagArray: processing <" .. inString .. ">", 30)
	end 

	if string.len(inString) < 1 then 
		trigger.action.outText("+++flagArray: empty flags", 30)
		return {} 
	end
	
	
	local flags = {}
	local rawElements = dcsCommon.splitString(inString, ",")
	-- go over all elements 
	for idx, anElement in pairs(rawElements) do 
		anElement = dcsCommon.trim(anElement)
		if dcsCommon.stringStartsWithDigit(anElement) and  dcsCommon.containsString(anElement, "-") then 
			-- interpret this as a range
			local theRange = dcsCommon.splitString(anElement, "-")
			local lowerBound = theRange[1]
			lowerBound = tonumber(lowerBound)
			local upperBound = theRange[2]
			upperBound = tonumber(upperBound)
			if lowerBound and upperBound then
				-- swap if wrong order
				if lowerBound > upperBound then 
					local temp = upperBound
					upperBound = lowerBound
					lowerBound = temp 
				end
				-- now add add numbers to flags
				for f=lowerBound, upperBound do 
					table.insert(flags, f)

				end
			else
				-- bounds illegal
				trigger.action.outText("+++flagArray: ignored range <" .. anElement .. "> (range)", 30)
			end
		else
			-- single number
			local f = dcsCommon.trim(anElement) -- DML flag upgrade: accept strings tonumber(anElement)
			if f then 
				table.insert(flags, f)

			else 
				trigger.action.outText("+++flagArray: ignored element <" .. anElement .. "> (single)", 30)
			end
		end
	end
	if verbose then 
		trigger.action.outText("+++flagArray: <" .. #flags .. "> flags total", 30)
	end 
	return flags
end

function dcsCommon.rangeArrayFromString(inString, verbose)
	if not verbose then verbose = false end 
	
	if verbose then 
		trigger.action.outText("+++rangeArray: processing <" .. inString .. ">", 30)
	end 

	if string.len(inString) < 1 then 
		trigger.action.outText("+++rangeArray: empty ranges", 30)
		return {} 
	end
	
	local ranges = {}
	local rawElements = dcsCommon.splitString(inString, ",")
	-- go over all elements 
	for idx, anElement in pairs(rawElements) do 
		anElement = dcsCommon.trim(anElement)
		local outRange = {}
		if dcsCommon.stringStartsWithDigit(anElement) and  dcsCommon.containsString(anElement, "-") then 
			-- interpret this as a range
			local theRange = dcsCommon.splitString(anElement, "-")
			local lowerBound = theRange[1]
			lowerBound = tonumber(lowerBound)
			local upperBound = theRange[2]
			upperBound = tonumber(upperBound)
			if lowerBound and upperBound then
				-- swap if wrong order
				if lowerBound > upperBound then 
					local temp = upperBound
					upperBound = lowerBound
					lowerBound = temp 
				end
				-- now add to ranges
				outRange[1] = lowerBound
				outRange[2] = upperBound
				table.insert(ranges, outRange)
				if verbose then 
					trigger.action.outText("+++rangeArray: new range <" .. lowerBound .. "> to <" .. upperBound .. ">", 30)
				end
			else
				-- bounds illegal
				trigger.action.outText("+++rangeArray: ignored range <" .. anElement .. "> (range)", 30)
			end
		else
			-- single number
			local f = dcsCommon.trim(anElement) 
			f = tonumber(f)
			if f then 
				outRange[1] = f
				outRange[2] = f
				table.insert(ranges, outRange)
				if verbose then 
					trigger.action.outText("+++rangeArray: new (single-val) range <" .. f .. "> to <" .. f .. ">", 30)
				end
			else 
				trigger.action.outText("+++rangeArray: ignored element <" .. anElement .. "> (single)", 30)
			end
		end
	end
	if verbose then 
		trigger.action.outText("+++rangeArray: <" .. #ranges .. "> ranges total", 30)
	end 
	return ranges
end

function dcsCommon.incFlag(flagName)
	local v = trigger.misc.getUserFlag(flagName)
	trigger.action.setUserFlag(flagName, v + 1)
end

function dcsCommon.decFlag(flagName)
	local v = trigger.misc.getUserFlag(flagName)
	trigger.action.setUserFlag(flagName, v - 1)
end

function dcsCommon.objectHandler(theObject, theCollector)
	table.insert(theCollector, theObject)
	return true 
end

function dcsCommon.getObjectsForCatAtPointWithRadius(aCat, thePoint, theRadius)
	if not aCat then aCat = Object.Category.UNIT end 
	local p = {x=thePoint.x, y=thePoint.y, z=thePoint.z}
	local collector = {}
	
	-- now build the search argument 
	local args = {
			id = world.VolumeType.SPHERE,
			params = {
				point = p,
				radius = theRadius
			}
		}
	
	-- now call search
	world.searchObjects(aCat, args, dcsCommon.objectHandler, collector)
	return collector
end

function dcsCommon.getSceneryObjectsInZone(theZone) -- DCS ZONE!!! 
	local aCat = 5 -- scenery
	-- WARNING: WE ARE USING DCS ZONES, NOT CFX!!!
	local p = {x=theZone.x, y=0, z=theZone.y}
	local lp = {x = p.x, y = p.z}
	p.y = land.getHeight(lp)
	local collector = {}
	
	-- now build the search argument 
	local args = {
			id = world.VolumeType.SPHERE,
			params = {
				point = p,
				radius = theZone.radius
			}
		}
	
	-- now call search
	world.searchObjects(aCat, args, dcsCommon.objectHandler, collector)
	return collector
end

function dcsCommon.getSceneryObjectInZoneByName(theName, theZone) -- DCS ZONE!!!
	local allObs = dcsCommon.getSceneryObjectsInZone(theZone)
	for idx, anObject in pairs(allObs) do 
		if tostring(anObject:getName()) == theName then return anObject end 
	end
	return nil 
end

--
-- bitwise operators
--
function dcsCommon.bitAND32(a, b)
	if not a then a = 0 end 
	if not b then b = 0 end 
	local z = 0
	local e = 1
	for i = 0, 31 do 
		local a1 = a % 2 -- 0 or 1
		local b1 = b % 2 -- 0 or 1
		if a1 == 1 and b1 == 1 then 
			a = a - 1 -- remove bit 
			b = b - 1 
			z = z + e
		else
			if a1 == 1 then a = a - 1 end -- remove bit 
			if b1 == 1 then b = b - 1 end 
		end
		a = a / 2 -- shift right
		b = b / 2		
		e = e * 2 -- raise e by 1 
	end
	return z
end

function dcsCommon.num2bin(a)
	if not a then a = 0 end 
	local z = ""
	for i = 0, 31 do 
		local a1 = a % 2 -- 0 or 1
		if a1 == 1 then 
			a = a - 1 -- remove bit 
			z = "1"..z
		else
			z = "0"..z
		end
		a = a / 2 -- shift right
	end
	return z
end

function dcsCommon.LSR(a, num)
	if not a then a = 0 end 
	if not num then num = 16 end 
	for i = 1, num do 
		local a1 = a % 2 -- 0 or 1
		if a1 == 1 then 
			a = a - 1 -- remove bit 
		end
		a = a / 2 -- shift right
	end
	return a
end

--
-- string wildcards 
--
function dcsCommon.processStringWildcards(inMsg)
	-- Replace STATIC bits of message like CR and zone name 
	if not inMsg then return "<nil inMsg>" end
	local formerType = type(inMsg)
	if formerType ~= "string" then inMsg = tostring(inMsg) end  
	if not inMsg then inMsg = "<inMsg is incompatible type " .. formerType .. ">" end 
	local outMsg = ""
	-- replace line feeds 
	outMsg = inMsg:gsub("<n>", "\n")

	return outMsg 
end

--
-- phonetic alphabet 
--
dcsCommon.alphabet = {
    a = "alpha",
    b = "bravo",
    c = "charlie",
    d = "delta",
    e = "echo",
    f = "foxtrot",
    g = "golf",
    h = "hotel",
    i = "india",
    j = "juliet",
    k = "kilo",
    l = "lima",
    m = "mike",
    n = "november",
    o = "oscar",
    p = "papa",
    q = "quebec",
    r = "romeo",
    s = "sierra",
    t = "tango",
    u = "uniform",
    v = "victor",
    w = "whiskey",
    x = "x-ray",
    y = "yankee",
    z = "zulu",
["0"] = "zero",
["1"] = "wun",
["2"] = "too",
["3"] = "tree",
["4"] = "fower",
["5"] = "fife" ,
["6"] = "six",
["7"] = "seven",
["8"] = "att",
["9"] = "niner",
[" "] = "break",
}

function dcsCommon.letter(inChar)
	local theChar = ""
	if type(inChar == "string") then 
		if #inChar < 1 then return "#ERROR0#" end
		inChar = string.lower(inChar)
		theChar = string.sub(inChar, 1, 1)
	elseif type(inChar == "number") then 
		if inChar > 255 then return "#ERROR>#" end 
		if inChar < 0 then return "#ERROR<#" end 
		theChar = char(inChar)
	else 
		return "#ERRORT#"
	end
--	trigger.action.outText("doing <" .. theChar .. ">", 30)
	local a = dcsCommon.alphabet[theChar]
	if a == nil then a = "#ERROR?#" end 
	return a 
end

function dcsCommon.spellString(inString)
	local res = ""
	local first = true 
	for i = 1, #inString do
		local c = inString:sub(i,i)
		if first then 
			res = dcsCommon.letter(c)
			first = false 
		else 
			res = res .. " " .. dcsCommon.letter(c)
		end
	end
	return res 
end

dcsCommon.letters = {"A", "B", "C", "D", "E", "F", "G", "H", "I", "J", "K", "L", "M", "N", 
"O", "P", "Q", "R", "S", "T", "U", "V", "W", "X", "Y", "Z", }
function dcsCommon.randomLetter(lowercase)
	local theLetter = dcsCommon.pickRandom(dcsCommon.letters)
	if lowercase then theLetter = string.lower(theLetter) end 
	return theLetter
end

--
-- RGBA from hex
--
function dcsCommon.hexString2RGBA(inString) 
	-- enter with "#FF0020" (RGB) or "#FF00AB99" RGBA
	-- check if it starts with #
	if not inString then return nil end 
	if #inString ~= 7 and #inString ~=9 then return nil end 
	if inString:sub(1, 1) ~= "#" then return nil end 
	inString = inString:lower()
	local red = tonumber("0x" .. inString:sub(2,3)) 
	if not red then red = 0 end 
	local green = tonumber("0x" .. inString:sub(4,5))
	if not green then green = 0 end 
	local blue = tonumber("0x" .. inString:sub(6,7))
	if not blue then blue = 0 end 
	local alpha = 255 
	if #inString == 9 then 
		alpha = tonumber("0x" .. inString:sub(8,9))
	end
	if not alpha then alpha = 0 end
	return {red/255, green/255, blue/255, alpha/255}
end


--
-- Player handling 
--
function dcsCommon.playerName2Coalition(playerName)
	if not playerName then return 0 end 
	local factions = {1,2}
	for idx, theFaction in pairs(factions) do 
		local players = coalition.getPlayers(theFaction)
		for idy, theUnit in pairs(players) do 
			local upName = theUnit:getPlayerName()
			if upName == playerName then return theFaction end
		end
	end
	return 0
end

function dcsCommon.isPlayerUnit(theUnit)
	-- new patch. simply check if getPlayerName returns something
	if not theUnit then return false end 
	if not Unit.isExist(theUnit) then return end 
	if not theUnit.getPlayerName then return false end -- map/static object 
	local pName = theUnit:getPlayerName()
	if pName then return true end 
	return false 
end

function dcsCommon.getPlayerUnit(name)
	for coa = 1, 2 do 
		local players = coalition.getPlayers(coa)
		for idx, theUnit in pairs(players) do 
			if theUnit:getPlayerName() == name then return theUnit end
		end
	end
	return nil 
end

--
-- theater and theater-related stuff 
--
function dcsCommon.getMapName()
	return env.mission.theatre
end

dcsCommon.magDecls = {Caucasus = 6.5,
					  MarianaIslands = 1,
					  Nevada = 12,
					  PersianGulf = 2,
					  Syria = 4,
					  Normandy = -12 -- 1944, -1 in 2016 
					  -- SinaiMap still missing 
					  -- Falklands still missing, big differences 
					  }
					  
function dcsCommon.getMagDeclForPoint(point) 
	-- WARNING! Approximations only, map-wide, not adjusted for year nor location!
	-- serves as a stub for the day when DCS provides correct info 
	local map = dcsCommon.getMapName()
	local decl = dcsCommon.magDecls[map]
	if not decl then 
		trigger.action.outText("+++dcsC: unknown map <" .. map .. ">, using dclenation 0", 30)
		decl = 0
	end
	return decl 
end 

--
-- iterators
--
-- iteratePlayers - call callback for all player units
-- callback is of signature callback(playerUnit)
--

function dcsCommon.iteratePlayers(callBack)
	local factions = {0, 1, 2}
	for idx, theFaction in pairs(factions) do 
		local players = coalition.getPlayers(theFaction)
		for idy, theUnit in pairs(players) do 
			callBack(theUnit)
		end
	end
end


--
-- MISC POINT CREATION
--
function dcsCommon.createPoint(x, y, z)
	local newPoint = {}
	newPoint.x = x
	newPoint.y = y
	newPoint.z = z -- works even if Z == nil
	return newPoint
end

function dcsCommon.copyPoint(inPoint) 
	local newPoint = {}
	newPoint.x = inPoint.x
	newPoint.y = inPoint.y
	-- handle xz only 
	if inPoint.z then 
		newPoint.z = inPoint.z 
	else 
		newPoint.z = inPoint.y 
	end
	return newPoint	
end

--
-- SEMAPHORES
--
dcsCommon.semaphores = {}

-- replacement for trigger.misc.getUserFlag
function dcsCommon.getUserFlag(flagName)
	if dcsCommon.semaphores[flagName] then 
		return dcsCommon.semaphores[flagName]
	end
	
	return trigger.misc.getUserFlag(flagName)
end

-- replacement for trigger.action.setUserFlag 
function dcsCommon.setUserFlag(flagName, theValue)
	-- not yet connected: semaphores
	
	-- forget semaphore content if new value is old-school 
	if type(theValue) == "number" then 
		dcsCommon.semaphores[theValue] = nil --return to old-school 
	end
	trigger.action.setUserFlag(flagName, theValue)
end

--
--
-- INIT
--
--
	-- init any variables, tables etc that the lib requires internally
	function dcsCommon.init()
		cbID = 0
		-- create ID tables
		dcsCommon.collectMissionIDs()
		
		--dcsCommon.uuIdent = 0
		if (dcsCommon.verbose) or true then
		  trigger.action.outText("dcsCommon v" .. dcsCommon.version .. " loaded", 10)
		end
	end

	
-- do init. 
dcsCommon.init()

cfxZones = {}
cfxZones.version = "4.1.2"

-- cf/x zone management module
-- reads dcs zones and makes them accessible and mutable 
-- by scripting.
--
-- Copyright (c) 2021 - 2023 by Christian Franz and cf/x AG
--

--[[-- VERSION HISTORY
- 4.0.0   - dmlZone OOP API started 
		  - code revision / refactoring 
		  - moved createPoint and copxPoint to dcsCommon, added bridging code 
		  - re-routed all createPoint() invocations to dcsCommon 
		  - removed anyPlayerInZone() because of cfxPlayer dependency
		  - numberArrayFromString() moved to dcsCommon, bridged 
		  - flagArrayFromString() moved to dcsCommon, bridged 
		  - doPollFlag() can differentiate between number method and string method 
		    to enable passing an immediate negative value 
		  - getNumberFromZoneProperty() enforces number return even on default
		  - immediate method switched to preceeding '#', to resolve conflict witzh 
		    negative numbers, backwards compatibility with old (dysfunctional) method 
- 4.0.1   - dmlZone:getName()
- 4.0.2   - removed verbosity from declutterZone (both versions)
- 4.0.3   - new processDynamicVZU()
	      - wildcard uses processDynamicVZU
- 4.0.4   - setFlagValue now supports multiple flags (OOP and classic)
		  - doSetFlagValue optimizations 
- 4.0.5   - dynamicAB wildcard 
		  - processDynamicValueVU
- 4.0.6   - hash mark forgotten QoL
- 4.0.7   - drawZone()
- 4.0.8   - markZoneWithObjects()
		  - cleanup 
		  - markCenterWithObject
		  - markPointWithObject
- 4.0.9   - createPolyZone now correctly returns new zone 
		  - createSimplePolyZone correctly passes location to createPolyZone 
		  - createPolyZone now correctly sets zone.point
		  - createPolyZone now correctly inits dcsOrigin
		  - createCircleZone noew correctly inits dcsOrigin
- 4.0.10  - getBoolFromZoneProperty also supports "on" (=true) and "off" (=false)
- 4.1.0   - getBoolFromZoneProperty 'on/off' support for dml variant as well 
- 4.1.1   - evalRemainder() updates 
- 4.1.2   - hash property missing warning 

--]]--

--
-- ====================
-- OOP dmlZone API HERE
-- ====================
--

dmlZone = {}
function dmlZone:new(o)
	o = o or {}
	setmetatable(o, self)
	self.__index = self 
	self.name = "dmlZone raw"
	self.isCircle = false
	self.isPoly = false
	self.radius = 0
	self.poly = {}
	self.bounds = {}
	self.properties = {}
	return o 
end 

--
-- CLASSIC INTERFACE
--
cfxZones.verbose = false
cfxZones.caseSensitiveProperties = false -- set to true to make property names case sensitive 
cfxZones.ups = 1 -- updates per second. updates moving zones

cfxZones.zones = {} -- these are the zone as retrieved from the mission.
					-- ALWAYS USE THESE, NEVER DCS's ZONES!!!!

function cfxZones.readFromDCS(clearfirst)
	if (clearfirst) then
		cfxZones.zones = {}
	end
	-- not all missions have triggers or zones
	if not env.mission.triggers then 
		if cfxZones.verbose then 
			trigger.action.outText("cf/x zones: no env.triggers defined", 10)
		end
		return
	end
	
	if not env.mission.triggers.zones then 
		if cfxZones.verbose then 
			trigger.action.outText("cf/x zones: no zones defined", 10)
		end
		return;
	end

	-- we only retrieve the data we need. At this point it is name, location and radius
	-- and put this in our own little  structure. we also convert to all upper case name for index
	-- and assume that the name may also carry meaning, e.g. 'LZ:' defines a landing zone
	-- so we can quickly create other sets from this
	-- zone object. DCS 2.7 introduced quads, so this is supported as well
	--   name - name in upper case
	--   isCircle - true if circular zone 
	--   isPoly - true if zone is defined by convex polygon, e.g. quad 
	--   point - vec3 (x 0 z) - zone's in-world center, used to place the coordinate
	--   radius - number, zero when quad
	--   bounds - aabb with attributes ul, ur, ll, lr (upper left .. lower right) as (x, 0, z)
	--   poly - array 1..n of poly points, wound counter-clockwise 
	
	for i, dcsZone in pairs(env.mission.triggers.zones) do
		if type(dcsZone) == 'table' then -- hint taken from MIST: verify type when reading from dcs
										 -- dcs data is like a box of chocolates...
			local newZone = dmlZone:new(nil) -- WAS: {} -- OOP introduction July 2023
			-- name, converted to upper is used only for indexing
			-- the original name remains untouched
			newZone.dcsZone = dcsZone
			newZone.name = dcsZone.name
			newZone.isCircle = false
			newZone.isPoly = false
			newZone.radius = 0
			newZone.poly = {}
			newZone.bounds = {}
			newZone.properties = {} -- dcs has this too, copy if present
			if dcsZone.properties then 
				newZone.properties = dcsZone.properties 
			else
				newZone.properties = {}
			end -- WARNING: REF COPY. May need to clone 
			
			local upperName = newZone.name:upper()
			
			-- location as 'point'
			-- WARNING: zones locs are 2D (x,y) pairs, while y in DCS is altitude.
			--          so we need to change (x,y) into (x, 0, z). Since Zones have no
			--          altitude (they are an infinite cylinder) this works. Remember to 
			--          drop y from zone calculations to see if inside. 
			-- WARNING: ME linked zones have a relative x any y 
			--          to the linked unit 
			if dcsZone.linkUnit then 
				-- calculate the zone's real position by accessing the unit's MX data 
				-- as precached by dcsCommon
				local ux, uy = dcsCommon.getUnitStartPosByID(dcsZone.linkUnit)
				newZone.point = dcsCommon.createPoint(ux + dcsZone.x, 0, uy + dcsZone.y)
				newZone.dcsOrigin = dcsCommon.createPoint(ux + dcsZone.x, 0, uy + dcsZone.y)
			else 
				newZone.point = dcsCommon.createPoint(dcsZone.x, 0, dcsZone.y)
				newZone.dcsOrigin = dcsCommon.createPoint(dcsZone.x, 0, dcsZone.y)
			end

			-- start type processing. if zone.type exists, we have a mission 
			-- created with 2.7 or above, else earlier 
			local zoneType = 0
			if (dcsZone.type) then 
				zoneType = dcsZone.type 
			end
			
			if zoneType == 0 then 
				-- circular zone 
				newZone.isCircle = true 
				newZone.radius = dcsZone.radius
				newZone.maxRadius = newZone.radius -- same for circular
	
			elseif zoneType == 2 then
				-- polyZone
				newZone.isPoly = true 
				newZone.radius = dcsZone.radius -- radius is still written in DCS, may change later. The radius has no meaning and is the last radius written before zone changed to poly.
				-- note that newZone.point is only inside the tone for 
				-- convex polys, and DML only correctly works with convex polys
				-- now transfer all point in the poly
				-- note: DCS in 2.7 misspells vertices as 'verticies'
				-- correct for this 
				newZone.maxRadius = 0
				local verts = {}
				if dcsZone.verticies then verts = dcsZone.verticies 
				else 
					-- in later versions, this was corrected
					verts = dcsZone.vertices -- see if this is ever called
				end
				
				for v=1, #verts do
					local dcsPoint = verts[v]
					local polyPoint = cfxZones.createPointFromDCSPoint(dcsPoint) -- (x, y) --> (x, 0, y-->z)
					newZone.poly[v] = polyPoint
					-- measure distance from zone's point, and store maxRadius 
					-- dcs always saves a point with the poly zone 
					local dist = dcsCommon.dist(newZone.point, polyPoint)
					if dist > newZone.maxRadius then newZone.maxRadius = dist end 
				end
			else 
				
				trigger.action.outText("cf/x zones: malformed zone #" .. i .. " unknown type " .. zoneType, 10)
			end
			

			-- calculate bounds
			cfxZones.calculateZoneBounds(newZone) 

			-- add to my table
			cfxZones.zones[upperName] = newZone -- WARNING: UPPER ZONE!!!
			--trigger.action.outText("znd: procced " .. newZone.name .. " with radius " .. newZone.radius, 30)
		else
			if cfxZones.verbose then 
				trigger.action.outText("cf/x zones: malformed zone #" .. i .. " dropped", 10)
			end
		end -- else var not a table
		
	end -- for all zones kvp
end -- readFromDCS

function cfxZones.calculateZoneBounds(theZone)
	if not (theZone) then return 
	end
	
	local bounds = theZone.bounds -- copy ref!
	
	if theZone.isCircle then 
		-- aabb are easy: center +/- radius 
		local center = theZone.point
		local radius = theZone.radius 
		-- dcs uses z+ is down on map
		-- upper left is center - radius 
		bounds.ul = dcsCommon.createPoint(center.x - radius, 0, center.z - radius)
		bounds.ur = dcsCommon.createPoint(center.x + radius, 0, center.z - radius)
		bounds.ll = dcsCommon.createPoint(center.x - radius, 0, center.z + radius)
		bounds.lr = dcsCommon.createPoint(center.x + radius, 0, center.z + radius)
		
	elseif theZone.isPoly then
		local poly = theZone.poly -- ref copy!
		-- create the four points
		local ll = cfxZones.createPointFromPoint(poly[1])
		local lr = cfxZones.createPointFromPoint(poly[1])
		local ul = cfxZones.createPointFromPoint(poly[1])
		local ur = cfxZones.createPointFromPoint(poly[1])
		
		local pRad = dcsCommon.dist(theZone.point, poly[1]) -- rRad is radius for polygon from theZone.point 
		
		-- now iterate through all points and adjust bounds accordingly 
		for v=2, #poly do 
			local vertex = poly[v]
			if (vertex.x < ll.x) then ll.x = vertex.x; ul.x = vertex.x end 
			if (vertex.x > lr.x) then lr.x = vertex.x; ur.x = vertex.x end 
			if (vertex.z < ul.z) then ul.z = vertex.z; ur.z = vertex.z end
			--if (vertex.z > ll.z) then ll.z = vertex.z; lr.z = vertex.z end
			if (vertex.z > ur.z) then ur.z = vertex.z; ul.z = vertex.z end 			
			local dp = dcsCommon.dist(theZone.point, vertex)
			if dp > pRad then pRad = dp end -- find largst distance to vertex
		end
		
		-- now keep the new point references
		-- and store them in the zone's bounds
		bounds.ll = ll
		bounds.lr = lr
		bounds.ul = ul
		bounds.ur = ur 
		-- we may need to ascertain why we need ul, ur, ll, lr instead of just ll and ur 
		-- store pRad 
		theZone.pRad = pRad -- not sure we'll ever need that, but at least we have it

	else 
		-- huston, we have a problem
		if cfxZones.verbose then 
			trigger.action.outText("cf/x zones: calc bounds: zone " .. theZone.name .. " has unknown type", 30)
		end
	end
	
end

function dmlZone:calculateZoneBounds()
	cfxZones.calculateZoneBounds(self)
end 

function cfxZones.createPoint(x, y, z)  -- bridge to dcsCommon, backward comp.
	return dcsCommon.createPoint(x, y, z) 
end

function cfxZones.copyPoint(inPoint) -- bridge to dcsCommon, backward comp.
	return dcsCommon.copyPoint(inPoint)
end

function cfxZones.createHeightCorrectedPoint(inPoint) -- this should be in dcsCommon
	local cP = dcsCommon.createPoint(inPoint.x, land.getHeight({x=inPoint.x, y=inPoint.z}),inPoint.z)
	return cP
end

function cfxZones.getHeightCorrectedZonePoint(theZone)
	local thePoint = cfxZone.getPoint(theZone)
	return cfxZones.createHeightCorrectedPoint(thePoint)
end

function dmlZone:getHeightCorrectedZonePoint()
	local thePoint = self:getPoint()
	return dcsCommon.createPoint(thePoint.x, land.getHeight({x=thePoint.x, y=thePoint.z}),thePoint.z)
end

function cfxZones.createPointFromPoint(inPoint)
	return cfxZones.copyPoint(inPoint)
end

function cfxZones.createPointFromDCSPoint(inPoint) 
	return dcsCommon.createPoint(inPoint.x, 0, inPoint.y)
end


function cfxZones.createRandomPointInsideBounds(bounds)
	-- warning: bounds do not move woth zone! may have to be updated
	local x = math.random(bounds.ll.x, ur.x)
	local z = math.random(bounds.ll.z, ur.z)
	return dcsCommon.createPoint(x, 0, z)
end

function cfxZones.createRandomPointOnZoneBoundary(theZone)
	if not theZone then return nil end 
	if theZone.isPoly then 
		local loc, dx, dy = cfxZones.createRandomPointInPolyZone(theZone, true)
		return loc, dx, dy 
	else 
		local loc, dx, dy = cfxZones.createRandomPointInCircleZone(theZone, true)
		return loc, dx, dy 
	end
end

function dmlZone:createRandomPointOnZoneBoundary()
	return cfxZones.createRandomPointOnZoneBoundary(self)
end 

function cfxZones.createRandomPointInZone(theZone)
	if not theZone then return nil end 
	if theZone.isPoly then 
		local loc, dx, dy = cfxZones.createRandomPointInPolyZone(theZone)
		return loc, dx, dy 
	else 
		local loc, dx, dy = cfxZones.createRandomPointInCircleZone(theZone)
		return loc, dx, dy 
	end
end

function dmlZone:createRandomPointInZone()
	local loc, dx, dy = cfxZones.createRandomPointInZone(self)
	return loc, dx, dy
end 


function cfxZones.randomPointInZone(theZone)
	local loc, dx, dy =  cfxZones.createRandomPointInZone(theZone)
	return loc, dx, dy 
end

function dmlZone:randomPointInZone()
	local loc, dx, dy =  cfxZones.createRandomPointInZone(self)
	return loc, dx, dy 
end

function cfxZones.createRandomPointInCircleZone(theZone, onEdge)
	if not theZone.isCircle then 
		trigger.action.outText("+++Zones: warning - createRandomPointInCircleZone called for non-circle zone <" .. theZone.name .. ">", 30)
		return {x=theZone.point.x, y=0, z=theZone.point.z}
	end
	
	-- ok, let's first create a random percentage value for the new radius
	-- now lets get a random degree
	local degrees = math.random() * 2 * 3.14152 -- radiants. 
	local r = theZone.radius 
	if not onEdge then 
		r = r * math.random()
	end 
	local p = cfxZones.getPoint(theZone) -- force update of zone if linked
	local dx = r * math.cos(degrees)
	local dz = r * math.sin(degrees)
	local px = p.x + dx -- r * math.cos(degrees)
	local pz = p.z + dz -- r * math.sin(degrees)
	return {x=px, y=0, z = pz}, dx, dz -- returns loc and offsets to theZone.point
end

function dmlZone:createRandomPointInCircleZone(theZone, onEdge)
	local p, dx, dz = cfxZones.createRandomPointInCircleZone(self, onEdge)
	return p, dx, dz 
end 

function cfxZones.createRandomPointInPolyZone(theZone, onEdge)
	if not theZone.isPoly then 
		trigger.action.outText("+++Zones: warning - createRandomPointInPolyZone called for non-poly zone <" .. theZone.name .. ">", 30)
		return dcsCommon.createPoint(theZone.point.x, 0, theZone.point.z)
	end
	-- force update of all points 
	local p = cfxZones.getPoint(theZone)
	
	-- point in convex poly: choose two different lines from that polygon 
	local lineIdxA = dcsCommon.smallRandom(#theZone.poly)
	repeat lineIdxB = dcsCommon.smallRandom(#theZone.poly) until (lineIdxA ~= lineIdxB)
	
	-- we now have two different lines. pick a random point on each. 
	-- we use lerp to pick any point between a and b 
	local a = theZone.poly[lineIdxA]
	lineIdxA = lineIdxA + 1 -- get next point in poly and wrap around
	if lineIdxA > #theZone.poly then lineIdxA = 1 end 
	local b = theZone.poly[lineIdxA] 
	local randompercent = math.random()
	local sourceA = dcsCommon.vLerp (a, b, randompercent)
	-- if all we want is a point on an edge, we are done 
	if onEdge then 
		local polyPoint = sourceA
		return polyPoint, polyPoint.x - p.x, polyPoint.z - p.z -- return loc, dx, dz 
	end 
	
	-- now get point on second line 
	a = theZone.poly[lineIdxB]
	lineIdxB = lineIdxB + 1 -- get next point in poly and wrap around
	if lineIdxB > #theZone.poly then lineIdxB = 1 end 
	b = theZone.poly[lineIdxB] 
	randompercent = math.random()
	local sourceB = dcsCommon.vLerp (a, b, randompercent)
	
	-- now take a random point on that line that entirely 
	-- runs through the poly 
	randompercent = math.random()
	local polyPoint = dcsCommon.vLerp (sourceA, sourceB, randompercent)
	return polyPoint, polyPoint.x - p.x, polyPoint.z - p.z -- return loc, dx, dz 
end

function dmlZone:createRandomPointInPolyZone(onEdge)
	local p, dx, dz = cfxZones.createRandomPointInPolyZone(self, onEdge)
	return p, dx, dz 
end 

function cfxZones.addZoneToManagedZones(theZone)
	local upperName = string.upper(theZone.name) -- newZone.name:upper()
	cfxZones.zones[upperName] = theZone
end

function dmlZone:addZoneToManagedZones()
	local upperName = string.upper(self.name) -- newZone.name:upper()
	cfxZones.zones[upperName] = self
end

function cfxZones.createUniqueZoneName(inName, searchSet)
	if not inName then return nil end 
	if not searchSet then searchSet = cfxZones.zones end 
	inName = inName:upper()
	while searchSet[inName] ~= nil do 
		inName = inName .. "X"
	end
	return inName
end

function cfxZones.createSimpleZone(name, location, radius, addToManaged)
	if not radius then radius = 10 end
	if not addToManaged then addToManaged = false end 
	if not location then 
		location = {}
	end
	if not location.x then location.x = 0 end 
	if not location.z then location.z = 0 end 
	
	local newZone = cfxZones.createCircleZone(name, location.x, location.z, radius)
	
	if addToManaged then 
		cfxZones.addZoneToManagedZones(newZone)
	end
	return newZone
end

function cfxZones.createCircleZone(name, x, z, radius) 
	local newZone = dmlZone:new(nil) -- {} OOP compatibility 
	newZone.isCircle = true
	newZone.isPoly = false
	newZone.poly = {}
	newZone.bounds = {}
			
	newZone.name = name
	newZone.radius = radius
	newZone.point = dcsCommon.createPoint(x, 0, z)
 	newZone.dcsOrigin = dcsCommon.createPoint(x, 0, z)

	-- props 
	newZone.properties = {}
	
	-- calculate my bounds
	cfxZones.calculateZoneBounds(newZone)
	
	return newZone
end

function cfxZones.createSimplePolyZone(name, location, points, addToManaged)
	if not addToManaged then addToManaged = false end 
	if not location then 
		location = {}
	end
	if not location.x then location.x = 0 end 
	if not location.z then location.z = 0 end 
	if not location.y then location.y = 0 end 

	local newZone = cfxZones.createPolyZone(name, points, location)
	
	if addToManaged then 
		cfxZones.addZoneToManagedZones(newZone)
	end
	return newZone
end

function cfxZones.createSimpleQuadZone(name, location, points, addToManaged)
	if not location then 
		location = {}
	end
	if not location.x then location.x = 0 end 
	if not location.z then location.z = 0 end 
		
	-- synthesize 4 points if they don't exist
	-- remember: in DCS positive x is up, positive z is right 
	if not points then 
		points = {} 
	end
	if not points[1] then 
		-- upper left 
		points[1] = {x = location.x-1, y = 0, z = location.z-1}
	end
	if not points[2] then 
		-- upper right 
		points[2] = {x = location.x-1, y = 0, z = location.z+1}
	end
	if not points[3] then 
		-- lower right 
		points[3] = {x = location.x+1, y = 0, z = location.z+1}
	end
	if not points[4] then 
		-- lower left 
		points[4] = {x = location.x+1, y = 0, z = location.z-1}
	end
	
	return cfxZones.createSimplePolyZone(name, location, points, addToManaged)
end

function cfxZones.createPolyZone(name, poly, location) -- poly must be array of point type
	local newZone = dmlZone:new(nil) -- {} OOP compatibility 
	if not location then location = {x=0, y=0, z=0} end 
	newZone.point = dcsCommon.createPoint(location.x, 0, location.z)
	newZone.dcsOrigin = dcsCommon.createPoint(location.x, 0, location.z)
	newZone.isCircle = false
	newZone.isPoly = true
	newZone.poly = {}
	newZone.bounds = {}
			
	newZone.name = name
	newZone.radius = 0
	-- copy poly
	for v=1, #poly do 
		local theVertex = poly[v] 
		newZone.poly[v] = cfxZones.createPointFromPoint(theVertex) 
	end
	
	-- properties 
	newZone.properties = {}
	
	cfxZones.calculateZoneBounds(newZone)
	return newZone 
end

function cfxZones.createRandomZoneInZone(name, inZone, targetRadius, entirelyInside)
	-- create a new circular zone with center placed inside inZone
	-- if entirelyInside is false, only the zone's center is guaranteed to be inside
	-- inZone.
	-- entirelyInside is not guaranteed for polyzones
		
	if inZone.isCircle then 
		local sourceRadius = inZone.radius
		if entirelyInside and targetRadius > sourceRadius then targetRadius = sourceRadius end
		if entirelyInside then sourceRadius = sourceRadius - targetRadius end
	
		-- ok, let's first create a random percentage value for the new radius
		local percent = 1 / math.random(100)
		-- now lets get a random degree
		local degrees = math.random(360) * 3.14152 / 180 -- ok, it's actually radiants. 
		local r = sourceRadius * percent 
		local x = inZone.point.x + r * math.cos(degrees)
		local z = inZone.point.z + r * math.sin(degrees)
		-- construct new zone
		local newZone = cfxZones.createCircleZone(name, x, z, targetRadius)
		return newZone
	
	elseif inZone.isPoly then 
		local newPoint = cfxZones.createRandomPointInPolyZone(inZone)
		-- construct new zone
		local newZone = cfxZones.createCircleZone(name, newPoint.x, newPoint.z, targetRadius)
		return newZone
		
	else 
		-- zone type unknown
		trigger.action.outText("CreateZoneInZone: unknown zone type for inZone =" .. inZone.name ,  10)
		return nil 
	end
end

-- polygon inside zone calculations


-- isleft returns true if point P is to the left of line AB 
-- by determining the sign (up or down) of the normal vector of 
-- the two vectors PA and PB in the y coordinate. We arbitrarily define
-- left as being > 0, so right is <= 0. As long as we always use the 
-- same comparison, it does not matter what up or down mean.
-- this is important because we don't know if dcs always winds quads
-- the same way, we must simply assume that they are wound as a polygon 
function cfxZones.isLeftXZ(A, B, P)
	return ((B.x - A.x)*(P.z - A.z) - (B.z - A.z)*(P.x - A.x)) > 0
end

-- returns true/false for inside
function cfxZones.isPointInsideQuad(thePoint, A, B, C, D) 
    -- Inside test (only convex polygons): 
	-- point lies on the same side of each quad's vertex AB, BC, CD, DA
	-- how do we find out which side a point lies on? via the cross product
	-- see isLeft below
	
	-- so all we need to do is make sure all results of isLeft for all
	-- four sides are the same
	mustMatch = isLeftXZ(A, B, thePoint) -- all test results must be the same and we are ok
									   -- they just must be the same side.
	if (cfxZones.isLeftXZ(B, C, thePoint ~= mustMatch)) then return false end -- on other side than all before
	if (cfxZones.isLeftXZ(C, D, thePoint ~= mustMatch)) then return false end 
	if (cfxZones.isLeftXZ(D, A, thePoint ~= mustMatch)) then return false end
	return true
end

-- generalized version of insideQuad, assumes winding of poly, poly convex, poly closed
function cfxZones.isPointInsidePoly(thePoint, poly)
	local mustMatch = cfxZones.isLeftXZ(poly[1], poly[2], thePoint)
	for v=2, #poly-1 do 
		if cfxZones.isLeftXZ(poly[v], poly[v+1], thePoint) ~= mustMatch then return false end
	end
	-- final test
	if cfxZones.isLeftXZ(poly[#poly], poly[1], thePoint) ~= mustMatch then return false end
	
	return true
end;

function cfxZones.isPointInsideZone(thePoint, theZone, radiusIncrease)
	-- radiusIncrease only works for circle zones 
	if not radiusIncrease then radiusIncrease = 0 end 
	local p = {x=thePoint.x, y = 0, z = thePoint.z} -- zones have no altitude
	if (theZone.isCircle) then 
		local zp = cfxZones.getPoint(theZone)
		local d = dcsCommon.dist(p, theZone.point)
		return d < theZone.radius + radiusIncrease, d 
	end 
	
	if (theZone.isPoly) then 
		--trigger.action.outText("zne: isPointInside: " .. theZone.name .. " is Polyzone!", 30)
		return (cfxZones.isPointInsidePoly(p, theZone.poly)), 0 -- always returns delta 0
	end

	trigger.action.outText("isPointInsideZone: Unknown zone type for " .. outerZone.name, 10)
end

function dmlZone:isPointInsideZone(thePoint, radiusIncrease) -- warning: param order!
	return cfxZones.isPointInsideZone(thePoint, self, radiusIncrease)
end 

-- isZoneInZone returns true if center of innerZone is inside  outerZone
function cfxZones.isZoneInsideZone(innerZone, outerZone) 
	local p = cfxZones.getPoint(innerZone)
	return cfxZones.isPointInsideZone(p, outerZone)	
end

function dmlZone:isZoneInsideZone(outerZone) 
	return cfxZones.isPointInsideZone(self:getPoint(), outerZone)	
end

function cfxZones.getZonesContainingPoint(thePoint, testZones) -- return array 
	if not testZones then 
		testZones = cfxZones.zones 
	end 
	
	local containerZones = {}
	for tName, tData in pairs(testZones) do 
		if cfxZones.isPointInsideZone(thePoint, tData) then 
			table.insert(containerZones, tData)
		end
	end

	return containerZones
end

function cfxZones.getFirstZoneContainingPoint(thePoint, testZones)
	if not testZones then 
		testZones = cfxZones.zones 
	end 
	
	for tName, tData in pairs(testZones) do 
		if cfxZones.isPointInsideZone(thePoint, tData) then 
			return tData
		end
	end

	return nil
end

function cfxZones.getAllZonesInsideZone(superZone, testZones) -- returnes array!
	if not testZones then 
		testZones = cfxZones.zones 
	end 
	
	local containedZones = {}
	for zName, zData in pairs(testZones) do
		if cfxZones.isZoneInsideZone(zData, superZone) then 
			if zData ~= superZone then 
				-- we filter superzone because superzone usually resides 
				-- inside itself 
				table.insert(containedZones, zData)
			end
		end
	end
	return containedZones 
end

function dmlZone:getAllZonesInsideZone(testZones)
	return cfxZones.getAllZonesInsideZone(self, testZones)
end


function cfxZones.getZonesWithAttributeNamed(attributeName, testZones)
	if not testZones then testZones = cfxZones.zones end 

	local attributZones = {}
	for aName,aZone in pairs(testZones) do
		local attr = cfxZones.getZoneProperty(aZone, attributeName)
		if attr then 
			-- this zone has the requested attribute
			table.insert(attributZones, aZone)
		end
	end
	return attributZones
end

--
-- zone volume management
--

function cfxZones.getZoneVolume(theZone)
	if not theZone then return nil end 
	
	if (theZone.isCircle) then 
		-- create a sphere volume
		local p = cfxZones.getPoint(theZone)
		p.y = land.getHeight({x = p.x, y = p.z})
		local r = theZone.radius
		if r < 10 then r = 10 end 
		local vol = {
			id = world.VolumeType.SPHERE,
			params = {
				point = p,
				radius = r
			}
		}
		return vol 
	elseif (theZone.isPoly) then 
		-- build the box volume, using the zone's bounds ll and ur points 
		local lowerLeft = {}
		-- we build x = westerm y = southern, Z = alt 
		local alt = land.getHeight({x=theZone.bounds.ll.x, y = theZone.bounds.ll.z}) - 10
		lowerLeft.x = theZone.bounds.ll.x 
		lowerLeft.z = theZone.bounds.ll.z 
		lowerLeft.y = alt -- we go lower 
		
		local upperRight = {}
		alt = land.getHeight({x=theZone.bounds.ur.x, y = theZone.bounds.ur.z}) + 10
		upperRight.x = theZone.bounds.ur.x 
		upperRight.z = theZone.bounds.ur.z 
		upperRight.y = alt -- we go higher 
		
		-- construct volume 
		local vol = {
			id = world.VolumeType.BOX,
			params = {
				min = lowerLeft,
				max = upperRight
			}
		}
		return vol 
	else 
		trigger.action.outText("zne: unknown zone type for <" .. theZone.name .. ">", 30)
	end
end

function dmlZone:getZoneVolume()
	return cfxZones.getZoneVolume(self)
end 


function cfxZones.declutterZone(theZone)
	if not theZone then return end 
	local theVol = cfxZones.getZoneVolume(theZone)
	world.removeJunk(theVol)
end

function dmlZone:declutterZone()
	local theVol = cfxZones.getZoneVolume(self)
	world.removeJunk(theVol)
end

--
-- units / groups in zone
--
function cfxZones.allGroupsInZone(theZone, categ) -- categ is optional, must be code 
	-- warning: does not check for existing!
	local inZones = {}
	local coals = {0, 1, 2} -- all coalitions
	for idx, coa in pairs(coals) do 
		local allGroups = coalition.getGroups(coa, categ)
		for key, group in pairs(allGroups) do -- iterate all groups
			if cfxZones.isGroupPartiallyInZone(group, theZone) then
				table.insert(inZones, group)
			end
		end
	end
	return inZones
end

function dmlZone:allGroupsInZone(categ)
	return cfxZones.allGroupsInZone(self, categ)
end

function cfxZones.allGroupNamesInZone(theZone, categ) -- categ is optional, must be code 
	-- warning: does not check for existing!
	local inZones = {}
	local coals = {0, 1, 2} -- all coalitions
	for idx, coa in pairs(coals) do 
		local allGroups = coalition.getGroups(coa, categ)
		for key, group in pairs(allGroups) do -- iterate all groups
			if cfxZones.isGroupPartiallyInZone(group, theZone) then
				table.insert(inZones, group:getName())
			end
		end
	end
	return inZones
end

function dmlZone:allGroupNamesInZone(categ)
	return cfxZones.allGroupNamesInZone(self, categ)
end

function cfxZones.allStaticsInZone(theZone, useOrigin) -- categ is optional, must be code 
	-- warning: does not check for existing!
	local inZones = {}
	local coals = {0, 1, 2} -- all coalitions
	for idx, coa in pairs(coals) do 
		local allStats = coalition.getStaticObjects(coa)
		for key, statO in pairs(allStats) do -- iterate all groups
			local oP = statO:getPoint()
			if useOrigin then 
				if cfxZones.pointInZone(oP, theZone, true) then 
					-- use DCS original coords
					table.insert(inZones, statO)
				end
			elseif cfxZones.pointInZone(oP, theZone) then
				table.insert(inZones, statO)
			end
		end
	end
	return inZones
end

function dmlZone:allStaticsInZone(useOrigin)
	return cfxZones.allStaticsInZone(self, useOrigin)
end


function cfxZones.groupsOfCoalitionPartiallyInZone(coal, theZone, categ) -- categ is optional
	local groupsInZone = {}
	local allGroups = coalition.getGroups(coal, categ)
	for key, group in pairs(allGroups) do -- iterate all groups
		if group:isExist() then
			if cfxZones.isGroupPartiallyInZone(group, theZone) then
				table.insert(groupsInZone, group)			
			end
		end
	end
	return groupsInZone
end

function cfxZones.isGroupPartiallyInZone(aGroup, aZone)
	if not aGroup then return false end 
	if not aZone then return false end 
		
	if not aGroup:isExist() then return false end 
	local allUnits = aGroup:getUnits()
	for uk, aUnit in pairs (allUnits) do 
		if aUnit:isExist() and aUnit:getLife() > 1 then 		
			local p = aUnit:getPoint()
			local inzone, percent, dist = cfxZones.pointInZone(p, aZone)
			if inzone then		
				return true
			end 
		end
	end
	return false
end

function cfxZones.isEntireGroupInZone(aGroup, aZone)
	if not aGroup then return false end 
	if not aZone then return false end 
	if not aGroup:isExist() then return false end 
	local allUnits = aGroup:getUnits()
	for uk, aUnit in pairs (allUnits) do 
		if aUnit:isExist() and aUnit:getLife() > 1 then 
			local p = aUnit:getPoint()
			if not cfxZones.isPointInsideZone(p, aZone) then 
				return false
			end
		end
	end
	return true
end

function dmlZone:isEntireGroupInZone(aGroup)
	return cfxZones.isEntireGroupInZone(aGroup, self)
end

--
-- Zone Manipulation
--

function cfxZones.offsetZone(theZone, dx, dz)
	-- first, update center 
	theZone.point.x = theZone.point.x + dx
	theZone.point.z = theZone.point.z + dz 
	
	-- now process all polygon points - it's empty for circular, so don't worry
	for v=1, #theZone.poly do 
		theZone.poly[v].x = theZone.poly[v].x + dx
		theZone.poly[v].z = theZone.poly[v].z + dz 
	end
	
	-- update zone bounds 
	theZone.bounds.ll.x = theZone.bounds.ll.x + dx 
	theZone.bounds.lr.x = theZone.bounds.lr.x + dx
	theZone.bounds.ul.x = theZone.bounds.ul.x + dx 
	theZone.bounds.ur.x = theZone.bounds.ur.x + dx

	theZone.bounds.ll.z = theZone.bounds.ll.z + dz 
	theZone.bounds.lr.z = theZone.bounds.lr.z + dz
	theZone.bounds.ul.z = theZone.bounds.ul.z + dz 
	theZone.bounds.ur.z = theZone.bounds.ur.z + dz
	
end

function dmlZone:offsetZone(dx, dz)
	cfxZones.offsetZone(self, dx, dz)
end


function cfxZones.moveZoneTo(theZone, x, z)
	local dx = x - theZone.point.x
	local dz = z - theZone.point.z 
	cfxZones.offsetZone(theZone, dx, dz)
end;

function dmlZone:moveZoneTo(x, z)
	cfxZones.moveZoneTo(self, x, z)
end

function cfxZones.centerZoneOnUnit(theZone, theUnit) 
	local thePoint = theUnit:getPoint()
	cfxZones.moveZoneTo(theZone, thePoint.x, thePoint.z)
end

function dmlZone:centerZoneOnUnit(theUnit) 
	local thePoint = theUnit:getPoint()
	self:moveZoneTo(thePoint.x, thePoint.z)
end


function cfxZones.dumpZones(zoneTable)
	if not zoneTable then zoneTable = cfxZones.zones end 
	
	trigger.action.outText("Zones START", 10)
	for i, zone in pairs(zoneTable) do 
		local myType = "unknown"
		if zone.isCircle then myType = "Circle" end
		if zone.isPoly then myType = "Poly" end 
		
		trigger.action.outText("#".. i .. ": " .. zone.name .. " of type " .. myType, 10)
	end
	trigger.action.outText("Zones end", 10)
end

function cfxZones.keysForTable(theTable)
	local keyset={}
	local n=0

	for k,v in pairs(tab) do
		n=n+1
		keyset[n]=k
	end
	return keyset
end


--
-- return all zones that have a specific named property
--
function cfxZones.zonesWithProperty(propertyName, searchSet)
	if not searchSet then searchSet = cfxZones.zones end 
	local theZones = {}
	for k, aZone in pairs(searchSet) do 
		if not aZone then 
			trigger.action.outText("+++zone: nil aZone for " .. k, 30)
		else 
			local lU = cfxZones.getZoneProperty(aZone, propertyName)
			if lU then 
				table.insert(theZones, aZone)
			end
		end
	end	
	return theZones
end

--
-- return all zones from the zone table that begin with string prefix
--
function cfxZones.zonesStartingWithName(prefix, searchSet)
	if not searchSet then searchSet = cfxZones.zones end 
	local prefixZones = {}
	prefix = prefix:upper() -- all zones have UPPERCASE NAMES! THEY SCREAM AT YOU
	for name, zone in pairs(searchSet) do
		if dcsCommon.stringStartsWith(name:upper(), prefix) then
			prefixZones[name] = zone -- note: ref copy!
		end
	end
	
	return prefixZones
end

--
-- return all zones from the zone table that begin with the string or set of strings passed in prefix 
-- if you pass 'true' as second (optional) parameter, it will first look for all zones that begin
-- with '+' and return only those. Use during debugging to force finding a specific zone
--
function cfxZones.zonesStartingWith(prefix, searchSet, debugging)
	-- you can force zones by having their name start with "+"
	-- which will force them to return immediately if debugging is true for this call

	if (debugging) then 
		local debugZones = cfxZones.zonesStartingWithName("+", searchSet)
		if not (next(debugZones) == nil) then -- # operator only works on array elements 
			--trigger.action.outText("returning zones with prefix <" .. prefix .. ">", 10)
			return debugZones 
		end 
	end
	
	if (type(prefix) == "string") then 
		return cfxZones.zonesStartingWithName(prefix, searchSet)
	end
	
	local allZones = {}
	for i=1, #prefix do 
		-- iterate through all names in prefix set
		local theName = prefix[i]
		local newZones = cfxZones.zonesStartingWithName(theName, searchSet)
		-- add them all to current table
		for zName, zInfo in pairs(newZones) do 
			allZones[zName] = zInfo -- will also replace doublets
		end
	end
	
	return allZones
end

function cfxZones.getZoneByName(aName, searchSet) 
	if not searchSet then searchSet = cfxZones.zones end 
	aName = aName:upper()
	return searchSet[aName] -- the joys of key value pairs
end

function cfxZones.getZonesContainingString(aString, searchSet) 
	if not searchSet then searchSet = cfxZones.zones end
	aString = string.upper(aString)
	resultSet = {}
	for zName, zData in pairs(searchSet) do 
		if aString == string.upper(zData.name) then 
			resultSet[zName] = zData
		end
	end
	
end;

-- filter zones by range to a point. returns indexed set
function cfxZones.getZonesInRange(point, range, theZones)
	if not theZones then theZones = cfxZones.zones end
	
	local inRangeSet = {}
	for zName, zData in pairs (theZones) do 
		if dcsCommon.dist(point, zData.point) < range then 
			table.insert(inRangeSet, zData)
		end
	end
	return inRangeSet 
end

-- get closest zone returns the zone that is closest to point 
function cfxZones.getClosestZone(point, theZones)
	if not theZones then theZones = cfxZones.zones end
	local lPoint = {x=point.x, y=0, z=point.z}
	local currDelta = math.huge 
	local closestZone = nil
	for zName, zData in pairs(theZones) do 
		local zPoint = cfxZones.getPoint(zData)
		local delta = dcsCommon.dist(lPoint, zPoint) -- emulate flag compare 
		if (delta < currDelta) then 
			currDelta = delta
			closestZone = zData
		end
	end
	return closestZone, currDelta 
end

-- return a random zone from the table passed in zones
function cfxZones.pickRandomZoneFrom(zones)
	if not zones then zones = cfxZones.zones end
	local indexedZones = dcsCommon.enumerateTable(zones)
	local r = math.random(#indexedZones)
	return indexedZones[r]
end

-- return an zone element by index 
function cfxZones.getZoneByIndex(theZones, theIndex) 
	local enumeratedZones = dcsCommon.enumerateTable(theZones)
	if (theIndex > #enumeratedZones) then
		trigger.action.outText("WARNING: zone index " .. theIndex .. " out of bounds - max = " .. #enumeratedZones, 30)
		return nil end
	if (theIndex < 1) then return nil end
	
	return enumeratedZones[theIndex]
end

-- place a smoke marker in center of zone, offset by dx, dy 
function cfxZones.markZoneWithSmoke(theZone, dx, dz, smokeColor, alt)
	if not alt then alt = 5 end 
	local point = cfxZones.getPoint(theZone) --{} -- theZone.point
	point.x = point.x + dx -- getpoint updates and returns copy 
	point.z = point.z + dz 
	-- get height at point 
	point.y = land.getHeight({x = point.x, y = point.z}) + alt
	-- height-correct
	--local newPoint= {x = point.x, y = land.getHeight({x = point.x, y = point.z}) + 3, z= point.z}
	trigger.action.smoke(point, smokeColor)
end

function dmlZone:markZoneWithSmoke(dx, dz, smokeColor, alt)
	cfxZones.markZoneWithSmoke(self, dx, dz, smokeColor, alt)
end

-- place a smoke marker in center of zone, offset by radius and degrees 
function cfxZones.markZoneWithSmokePolar(theZone, radius, degrees, smokeColor, alt)
	local rads = degrees * math.pi / 180
	local dx = radius * math.sin(rads)
	local dz = radius * math.cos(rads)
	cfxZones.markZoneWithSmoke(theZone, dx, dz, smokeColor, alt)
end

function dmlZone:markZoneWithSmokePolar(radius, degrees, smokeColor, alt)
	cfxZones.markZoneWithSmokePolar(self, radius, degrees, smokeColor, alt)
end

-- place a smoke marker in center of zone, offset by radius and randomized degrees 
function cfxZones.markZoneWithSmokePolarRandom(theZone, radius, smokeColor)
	local degrees = math.random(360)
	cfxZones.markZoneWithSmokePolar(theZone, radius, degrees, smokeColor)
end

function dmlZone:markZoneWithSmokePolarRandom(radius, smokeColor)
	local degrees = math.random(360)
	self:markZoneWithSmokePolar(radius, degrees, smokeColor)
end

function cfxZones.pointInOneOfZones(thePoint, zoneArray, useOrig) 
	if not zoneArray then zoneArray = cfxZones.zones end 
	for idx, theZone in pairs(zoneArray) do 
		local isIn, percent, dist = cfxZones.pointInZone(thePoint, theZone, useOrig)
		if isIn then return isIn, percent, dist, theZone end 
	end
	return false, 0, 0, nil 
end


-- unitInZone returns true if theUnit is inside the zone 
-- the second value returned is the percentage of distance
-- from center to rim, with 100% being entirely in center, 0 = outside
-- the third value returned is the distance to center
function cfxZones.pointInZone(thePoint, theZone, useOrig)

	if not (theZone) then return false, 0, 0 end
		
	local pflat = {x = thePoint.x, y = 0, z = thePoint.z}
	
	local zpoint 
	if useOrig then
		zpoint = cfxZones.getDCSOrigin(theZone)
	else 
		zpoint = cfxZones.getPoint(theZone) -- updates zone if linked 
	end
	local ppoint = thePoint -- xyz
	local pflat = {x = ppoint.x, y = 0, z = ppoint.z}
	local dist = dcsCommon.dist(zpoint, pflat)
	
	if theZone.isCircle then 
		if theZone.radius <= 0 then 
			return false, 0, 0
		end

		local success = dist < theZone.radius
		local percentage = 0
		if (success) then 
			percentage = 1 - dist / theZone.radius 
		end
		return success, percentage, dist 
	
	elseif theZone.isPoly then
		local success = cfxZones.isPointInsidePoly(pflat, theZone.poly)
		return success, 0, dist
	else 
		trigger.action.outText("pointInZone: Unknown zone type for " .. theZone.name, 10)
	end

	return false
end

function dmlZone:pointInZone(thePoint, useOrig)
	return cfxZones.pointInZone(thePoint, self, useOrig)
end


function cfxZones.unitInZone(theUnit, theZone)
	if not (theUnit) then return false, 0, 0 end
	if not (theUnit:isExist()) then return false, 0, 0 end
	-- force zone update if it is linked to another zone 
	-- pointInZone does update
	local thePoint = theUnit:getPoint()
	return cfxZones.pointInZone(thePoint, theZone)
end

function dmlZone:unitInZone(theUnit)
	if not (theUnit) then return false, 0, 0 end
	if not (theUnit:isExist()) then return false, 0, 0 end
	-- force zone update if it is linked to another zone 
	-- pointInZone does update
	local thePoint = theUnit:getPoint()
	return self:pointInZone(thePoint)
end

-- returns all units of the input set that are inside the zone 
function cfxZones.unitsInZone(theUnits, theZone)
	if not theUnits then return {} end
	if not theZone then return {} end
	
	local zoneUnits = {}
	for index, aUnit in pairs(theUnits) do 
		if cfxZones.unitInZone(aUnit, theZone) then 
			table.insert( zoneUnits, aUnit)
		end
	end
	return zoneUnits
end

function dmlZone:unitsInZone(theUnits)
	if not theUnits then return {} end
	local zoneUnits = {}
	for index, aUnit in pairs(theUnits) do 
		if self:unitInZone(aUnit) then 
			table.insert(zoneUnits, aUnit)
		end
	end
	return zoneUnits
end

function cfxZones.closestUnitToZoneCenter(theUnits, theZone)
	-- does not care if they really are in zone. call unitsInZone first
	-- if you need to have them filtered
	-- theUnits MUST BE ARRAY
	if not theUnits then return nil end
	if #theUnits == 0 then return nil end
	local closestUnit = theUnits[1]
	local zP = cfxZones.getPoint(theZone)
	local smallestDist = math.huge
	for i=2, #theUnits do
		local aUnit = theUnits[i]
		local currDist = dcsCommon.dist(zP, aUnit:getPoint())
		if smallestDist > currDelta then 
			closestUnit = aUnit
			smallestDist = currDist
		end
	end
	return closestUnit
end

function dmlZone:closestUnitToZoneCenter(theUnits)
	return cfxZones.closestUnitToZoneCenter(theUnits, self)
end

-- grow zone
function cfxZones.growZone()
	-- circular zones simply increase radius
	-- poly zones: not defined 
	
end


-- creating units in a zone
function cfxZones.createGroundUnitsInZoneForCoalition (theCoalition, groupName, theZone, theUnits, formation, heading, liveries) 
	-- theUnits can be string or table of string 
	if not groupName then groupName = "G_"..theZone.name end 
	-- group name will be taken from zone name and prependend with "G_"
	local theGroup = dcsCommon.createGroundGroupWithUnits(groupName, theUnits, theZone.radius, nil, formation, nil, liveries)
	
	-- turn the entire formation to heading
	if (not heading) then heading = 0 end
	dcsCommon.rotateGroupData(theGroup, heading) -- currently, group is still at origin, no cx, cy
	
	
	-- now move the group to center of theZone
	dcsCommon.moveGroupDataTo(theGroup, 
						  theZone.point.x, 
						  theZone.point.z) -- watchit: Z!!!

	-- create the group in the world and return it
	-- first we need to translate the coalition to a legal 
	-- country. we use UN for neutral, cjtf for red and blue 
	local theSideCJTF = dcsCommon.coalition2county(theCoalition)
	-- store cty and cat for later access. DCS doesn't need it, but we may 
	
	theGroup.cty = theSideCJTF
	theGroup.cat = Group.Category.GROUND
	
    -- create a copy of the group data for 
	-- later reference 
	local groupDataCopy = dcsCommon.clone(theGroup)

	local newGroup = coalition.addGroup(theSideCJTF, Group.Category.GROUND, theGroup)
	return newGroup, groupDataCopy
end

--
-- ===============
-- FLAG PROCESSING 
-- ===============
--

--
-- Flag Pulling 
--
function cfxZones.pulseFlag(theFlag, method, theZone)
	local args = {}
	args.theFlag = theFlag
	args.method = method
	args.theZone = theZone 
	local delay = 3
	if dcsCommon.containsString(method, ",") then 
		local parts = dcsCommon.splitString(method, ",")
		delay = parts[2]
		if delay then delay = tonumber(delay) end  
	end
	if not delay then delay = 3 end 
	if theZone.verbose then 
		trigger.action.outText("+++zne: RAISING pulse t="..delay.." for flag <" .. theFlag .. "> in zone <" .. theZone.name ..">", 30)
	end 
	local newVal = 1
	cfxZones.setFlagValue(theFlag, newVal, theZone)
	
	-- schedule second half of pulse 
	timer.scheduleFunction(cfxZones.unPulseFlag, args, timer.getTime() + delay)
end

function dmlZone:pulseFlag(theFlag, method)
	cfxZones.pulseFlag(theFlag, method, self)
end

function cfxZones.unPulseFlag(args)
	local theZone = args.theZone
	local method = args.method 
	local theFlag = args.theFlag 
	local newVal = 0
	-- we may later use method to determine pulse direction / newVal
	-- for now, we always go low 
	if theZone.verbose then 
		trigger.action.outText("+++zne: DOWNPULSE pulse for flag <" .. theFlag .. "> in zone <" .. theZone.name ..">", 30)
	end
	cfxZones.setFlagValue(theFlag, newVal, theZone)
end

function cfxZones.evalRemainder(remainder, theZone)
	local rNum = tonumber(remainder)
	if not rNum then 
		-- we use remainder as name for flag 
		-- PROCESS ESCAPE SEQUENCES
		local esc = string.sub(remainder, 1, 1)
		local last = string.sub(remainder, -1)
		if esc == "@" then 
			remainder = string.sub(remainder, 2)
			remainder = dcsCommon.trim(remainder)
		end
		
		if esc == "(" and last == ")" and string.len(remainder) > 2 then 
			-- note: iisues with startswith("(") ???
			remainder = string.sub(remainder, 2, -2)
			remainder = dcsCommon.trim(remainder)		
		end
		if esc == "\"" and last == "\"" and string.len(remainder) > 2 then 
			remainder = string.sub(remainder, 2, -2)
			remainder = dcsCommon.trim(remainder)		
		end
		if cfxZones.verbose then 
			trigger.action.outText("+++zne: accessing flag <" .. remainder .. ">", 30)
		end 
		rNum = cfxZones.getFlagValue(remainder, theZone)
	end 
	return rNum
end

function dmlZone:evalRemainder(remainder)
	return cfxZones.evalRemainder(remainder, self)
end

function cfxZones.doPollFlag(theFlag, method, theZone) -- no OOP equivalent
	-- WARNING: 
	-- if method is a number string, it will be interpreted as follows:
	-- positive number: set immediate 
	-- negative: decrement by amouint
	if not theZone then 
		trigger.action.outText("+++zones: nil theZone on pollFlag", 30)
	end

	local mt = type(method)
	if mt == "number" then 
		method = "#" .. method -- convert to immediate 
		mt = "string"
	elseif mt ~= "string" then 
		trigger.action.outText("+++zne: warning: zone <" .. theZone.name .. "> method type <" .. mt .. "> received. Ignoring", 30)
		return 
	end

	local val = nil
	method = method:lower()
	method = dcsCommon.trim(method)
	val = tonumber(method) -- see if val can be directly converted 
	if dcsCommon.stringStartsWith(method, "+") or 
	   dcsCommon.stringStartsWith(method, "-") 
	then 
		-- skip this processing, a legal method can start with "+" or "-"
		-- and we interpret it as a method to increase or decrease by amount
	elseif (val ~= nil) then 
		-- provision to handle direct (positive) numbers (legacy support)
		-- method can be converted to number but does not start with - or +
		-- since all negative numbers start with '-' above guard will skip, positive will end up here
		cfxZones.setFlagValue(theFlag, val, theZone)
		if cfxZones.verbose or theZone.verbose then
			trigger.action.outText("+++zones: flag <" .. theFlag .. "> changed to #" .. val, 30)
		end 
		return
	else 
	end

	if dcsCommon.stringStartsWith(method, "#") then 
		-- immediate value command. remove # and eval remainder 
		local remainder = dcsCommon.removePrefix(method, "#")
		val = cfxZones.evalRemainder(remainder) -- always returens a number
		cfxZones.setFlagValue(theFlag, val, theZone)
		if theZone.verbose then 
			trigger.action.outText("+++zones: poll setting immediate <" .. theFlag .. "> in <" .. theZone.name .. "> to <" .. val .. ">", 30)
		end
		return 
	end
	
	local currVal = cfxZones.getFlagValue(theFlag, theZone)
	if method == "inc" or method == "f+1" then 
		--trigger.action.setUserFlag(theFlag, currVal + 1)
		cfxZones.setFlagValue(theFlag, currVal+1, theZone)
		
	elseif method == "dec" or method == "f-1" then 
		-- trigger.action.setUserFlag(theFlag, currVal - 1)
		cfxZones.setFlagValue(theFlag, currVal-1, theZone)

	elseif method == "off" or method == "f=0" then 
		-- trigger.action.setUserFlag(theFlag, 0)
		cfxZones.setFlagValue(theFlag, 0, theZone)

	elseif method == "flip" or method == "xor" then 
		if currVal ~= 0 then 
--			trigger.action.setUserFlag(theFlag, 0)
			cfxZones.setFlagValue(theFlag, 0, theZone)

		else 
			--trigger.action.setUserFlag(theFlag, 1)
			cfxZones.setFlagValue(theFlag, 1, theZone)
		end
		
	elseif dcsCommon.stringStartsWith(method, "pulse") then 
		cfxZones.pulseFlag(theFlag, method, theZone)
		
	elseif dcsCommon.stringStartsWith(method, "+") then 
		-- we add whatever is to the right 
		local remainder = dcsCommon.removePrefix(method, "+")
		local adder = cfxZones.evalRemainder(remainder)
		cfxZones.setFlagValue(theFlag, currVal+adder, theZone)
		if theZone.verbose then 
			trigger.action.outText("+++zones: (poll) updating with '+' flag <" .. theFlag .. "> in <" .. theZone.name .. "> by <" .. adder .. "> to <" .. adder + currVal .. ">", 30)
		end
		
	elseif dcsCommon.stringStartsWith(method, "-") then 
		-- we subtract whatever is to the right 
		local remainder = dcsCommon.removePrefix(method, "-")
		local adder = cfxZones.evalRemainder(remainder)
		cfxZones.setFlagValue(theFlag, currVal-adder, theZone)

	else 
		if method ~= "on" and method ~= "f=1" then 
			trigger.action.outText("+++zones: unknown method <" .. method .. "> - using 'on'", 30)
		end
		-- default: on.
--		trigger.action.setUserFlag(theFlag, 1)
		cfxZones.setFlagValue(theFlag, 1, theZone)
	end
	
	if cfxZones.verbose then
		local newVal = cfxZones.getFlagValue(theFlag, theZone)
		trigger.action.outText("+++zones: flag <" .. theFlag .. "> changed from " .. currVal .. " to " .. newVal, 30)
	end 
end

function cfxZones.pollFlag(theFlag, method, theZone) 
	local allFlags = {}
	if dcsCommon.containsString(theFlag, ",") then 
		if cfxZones.verbose then 
			trigger.action.outText("+++zones: will poll flag set <" .. theFlag .. "> with " .. method, 30)
		end
		allFlags = dcsCommon.splitString(theFlag, ",")
	else 
		table.insert(allFlags, theFlag)
	end
	
	for idx, aFlag in pairs(allFlags) do 
		aFlag = dcsCommon.trim(aFlag)
		-- note: mey require range preprocessing, but that's not
		-- a priority 
		cfxZones.doPollFlag(aFlag, method, theZone)
	end 
end

function dmlZone:pollFlag(theFlag, method)
	cfxZones.pollFlag(theFlag, method, self)
end

function cfxZones.expandFlagName(theFlag, theZone) 
	if not theFlag then return "!NIL" end 
	local zoneName = "<dummy>"
	if theZone then 
		zoneName = theZone.name -- for flag wildcards
	end
	
	if type(theFlag) == "number" then 
		-- straight number, return 
		return theFlag
	end
	
	-- we assume it's a string now
	theFlag = dcsCommon.trim(theFlag) -- clear leading/trailing spaces
	local nFlag = tonumber(theFlag) 
	if nFlag then -- a number, legal
		return theFlag
	end
		
	-- now do wildcard processing. we have alphanumeric
	if dcsCommon.stringStartsWith(theFlag, "*") then  
		theFlag = zoneName .. theFlag
	end
	return theFlag
end

function dmlZone:setFlagValue(theFlag, theValue)
	cfxZones.setFlagValueMult(theFlag, theValue, self)
end

function cfxZones.setFlagValue(theFlag, theValue, theZone)
	cfxZones.setFlagValueMult(theFlag, theValue, theZone)
end

function cfxZones.setFlagValueMult(theFlag, theValue, theZone)
	local allFlags = {}
	if dcsCommon.containsString(theFlag, ",") then 
		if cfxZones.verbose then 
			trigger.action.outText("+++zones: will multi-set flags <" .. theFlag .. "> to " .. theValue, 30)
		end
		allFlags = dcsCommon.splitString(theFlag, ",")
	else 
		table.insert(allFlags, theFlag)
	end
	
	for idx, aFlag in pairs(allFlags) do 
		aFlag = dcsCommon.trim(aFlag)
		-- note: mey require range preprocessing, but that's not
		-- a priority 
		cfxZones.doSetFlagValue(aFlag, theValue, theZone)
	end 
end

function cfxZones.doSetFlagValue(theFlag, theValue, theZone)
	local zoneName = "<dummy>"
	if not theZone then 
		trigger.action.outText("+++Zne: no zone on setFlagValue", 30) -- mod me for detector
	else 
		zoneName = theZone.name -- for flag wildcards
	end
	
	if type(theFlag) == "number" then 
		-- straight set, oldschool ME flag 
		trigger.action.setUserFlag(theFlag, theValue)
		return 
	end
	
	-- we assume it's a string now
	theFlag = dcsCommon.trim(theFlag) -- clear leading/trailing spaces	
	-- some QoL: detect "<none>"
	if dcsCommon.containsString(theFlag, "<none>") then 
		trigger.action.outText("+++Zone: warning - setFlag has '<none>' flag name in zone <" .. zoneName .. ">", 30) -- if error, intended break
	end
	
	-- now do wildcard processing. we have alphanumeric
	if dcsCommon.stringStartsWith(theFlag, "*") then  
		theFlag = zoneName .. theFlag
	end
	trigger.action.setUserFlag(theFlag, theValue)
end 



function cfxZones.getFlagValue(theFlag, theZone)
	local zoneName = "<dummy>"
	if not theZone or not theZone.name then 
		trigger.action.outText("+++Zne: no zone or zone name on getFlagValue", 30)
	else 
		zoneName = theZone.name -- for flag wildcards
	end
	
	if type(theFlag) == "number" then 
		-- straight get, ME flag 
		return tonumber(trigger.misc.getUserFlag(theFlag))
	end
	
	-- we assume it's a string now
	theFlag = dcsCommon.trim(theFlag) -- clear leading/trailing spaces
	local nFlag = tonumber(theFlag) 
	if nFlag then 
		return tonumber(trigger.misc.getUserFlag(theFlag))
	end
	
	-- some QoL: detect "<none>"
	if dcsCommon.containsString(theFlag, "<none>") then 
		trigger.action.outText("+++Zone: warning - getFlag has '<none>' flag name in zone <" .. zoneName .. ">", 30) -- break here
	end
	
	-- now do wildcard processing. we have alphanumeric
	if dcsCommon.stringStartsWith(theFlag, "*") then  
			theFlag = zoneName .. theFlag
	end
	return tonumber(trigger.misc.getUserFlag(theFlag))
end

function dmlZone:getFlagValue(theFlag)
	return cfxZones.getFlagValue(theFlag, self)
end

function cfxZones.verifyMethod(theMethod, theZone)
	local lMethod = string.lower(theMethod)
	if lMethod == "#" or lMethod == "change" then 
		return true
	end

	if lMethod == "0" or lMethod == "no" or lMethod == "false" 
	   or lMethod == "off" then 
		return true  
	end
	
	if lMethod == "1" or lMethod == "yes" or lMethod == "true" 
	   or lMethod == "on" then 
	    return true  
	end
	
	if lMethod == "inc" or lMethod == "+1" then 
		return true
	end
	
	if lMethod == "dec" or lMethod == "-1" then 
		return true 
	end 
	
	if lMethod == "lohi" or lMethod == "pulse" then 
		return true
	end
	
	if lMethod == "hilo" then 
		return true
	end
	
	-- number constraints
	-- or flag constraints 	-- ONLY RETURN TRUE IF CHANGE AND CONSTRAINT MET 
	local op = string.sub(theMethod, 1, 1) 
	local remainder = string.sub(theMethod, 2)
	remainder = dcsCommon.trim(remainder) -- remove all leading and trailing spaces

	if true then 
		-- we have a comparison = ">", "=", "<" followed by a number 
		-- THEY TRIGGER EACH TIME lastVal <> currVal AND condition IS MET  
		if op == "=" then 
			return true
		end
		
		if op == "#" or op == "~" then 
			return true
		end 
		
		if op == "<" then 
			return true
		end
		
		if op == ">" then 
			return true
		end
	end
	
	return false 
end

function dmlZone:verifyMethod(theMethod)
	return cfxZones.verifyMethod(theMethod, self)
end

-- method-based flag testing 
function cfxZones.evalFlagMethodImmediate(currVal, theMethod, theZone)
	-- immediate eval - does not look at last val. 
	-- return true/false/value based on theMethod's contraints 
	-- simple constraints
	local lMethod = string.lower(theMethod)
	if lMethod == "#" or lMethod == "change" then 
		-- ALWAYS RETURNS TRUE for currval <> 0, flase if currval = 0
		return currVal ~= 0  
	end
	
	if lMethod == "0" or lMethod == "no" or lMethod == "false" 
	   or lMethod == "off" then 
		-- WARNING: ALWAYS RETURNS FALSE
		return false  
	end
	
	if lMethod == "1" or lMethod == "yes" or lMethod == "true" 
	   or lMethod == "on" then 
	    -- WARNING: ALWAYS RETURNS TRUE
		return true  
	end
	
	if lMethod == "inc" or lMethod == "+1" then 
		return currVal+1 -- this may be unexpected
	end
	
	if lMethod == "dec" or lMethod == "-1" then 
		return currVal-1 -- this may be unexpectd
	end 
	
	-- number constraints
	-- or flag constraints 
	-- ONLY RETURN TRUE IF CHANGE AND CONSTRAINT MET 
	local op = string.sub(theMethod, 1, 1) 
	local remainder = string.sub(theMethod, 2)
	remainder = dcsCommon.trim(remainder) -- remove all leading and trailing spaces
	local rNum = tonumber(remainder)
	if not rNum then 
		-- we use remainder as name for flag 
		-- PROCESS ESCAPE SEQUENCES
		local esc = string.sub(remainder, 1, 1)
		local last = string.sub(remainder, -1)
		if esc == "@" then 
			remainder = string.sub(remainder, 2)
			remainder = dcsCommon.trim(remainder)
		end
		
		if esc == "(" and last == ")" and string.len(remainder) > 2 then 
			-- note: iisues with startswith("(") ???
			remainder = string.sub(remainder, 2, -2)
			remainder = dcsCommon.trim(remainder)		
		end
		if esc == "\"" and last == "\"" and string.len(remainder) > 2 then 
			remainder = string.sub(remainder, 2, -2)
			remainder = dcsCommon.trim(remainder)		
		end
		if cfxZones.verbose then 
			trigger.action.outText("+++zne: accessing flag <" .. remainder .. ">", 30)
		end 
		rNum = cfxZones.getFlagValue(remainder, theZone)
	end 
	if rNum then 
		-- we have a comparison = ">", "=", "<" followed by a number  
		if op == "=" then 
			return currVal == rNum
		end
		
		if op == "#" or op == "~" then 
			return currVal ~= rNum 
		end 
		
		if op == "<" then 
			return currVal < rNum
		end
		
		if op == ">" then 
			return currVal > rNum
		end
	end
	
	-- if we get here, we have an error 
	local zoneName = "<NIL>"
	if theZone then zoneName = theZone.name end 
	trigger.action.outText("+++Zne: illegal |" .. theMethod .. "| in eval for zone " .. zoneName, 30 )
	return false 	
end

function dmlZone:evalFlagMethodImmediate(currVal, theMethod, theZone)
	return cfxZones.evalFlagMethodImmediate(currVal, theMethod, self)
end


function cfxZones.testFlagByMethodForZone(currVal, lastVal, theMethod, theZone)
	-- return true/false based on theMethod's contraints 
	-- simple constraints
	-- ONLY RETURN TRUE IF CHANGE AND CONSTRAINT MET 
	local lMethod = string.lower(theMethod)
	if lMethod == "#" or lMethod == "change" then 
		-- check if currVal different from lastVal
		return currVal ~= lastVal  
	end
	
	if lMethod == "0" or lMethod == "no" or lMethod == "false" 
	   or lMethod == "off" then 
		-- WARNING: ONLY RETURNS TRUE IF FALSE AND lastval not zero!
		return currVal == 0 and currVal ~= lastVal  
	end
	
	if lMethod == "1" or lMethod == "yes" or lMethod == "true" 
	   or lMethod == "on" then 
	    -- WARNING: only returns true if lastval was false!!!!
		return (currVal ~= 0 and lastVal == 0)  
	end
	
	if lMethod == "inc" or lMethod == "+1" then 
--		return currVal == lastVal+1 -- better: test for greater than 
		return currVal > lastVal
	end
	
	if lMethod == "dec" or lMethod == "-1" then 
		--return currVal == lastVal-1
		return currVal < lastVal 
	end 
	
	if lMethod == "lohi" or lMethod == "pulse" then 
		return (lastVal <= 0 and currVal > 0)
	end
	
	if lMethod == "hilo" then 
		return (lastVal > 0 and currVal <= 0)
	end
	
	-- number constraints
	-- or flag constraints 
	-- ONLY RETURN TRUE IF CHANGE AND CONSTRAINT MET 
	local op = string.sub(theMethod, 1, 1) 
	local remainder = string.sub(theMethod, 2)
	remainder = dcsCommon.trim(remainder) -- remove all leading and trailing spaces
	local rNum = tonumber(remainder)
	if not rNum then 
		-- we use remainder as name for flag 
		-- PROCESS ESCAPE SEQUENCES
		local esc = string.sub(remainder, 1, 1)
		local last = string.sub(remainder, -1)
		if esc == "@" then 
			remainder = string.sub(remainder, 2)
			remainder = dcsCommon.trim(remainder)
		end
		
		if esc == "(" and last == ")" and string.len(remainder) > 2 then 
			-- note: iisues with startswith("(") ???
			remainder = string.sub(remainder, 2, -2)
			remainder = dcsCommon.trim(remainder)		
		end
		if esc == "\"" and last == "\"" and string.len(remainder) > 2 then 
			remainder = string.sub(remainder, 2, -2)
			remainder = dcsCommon.trim(remainder)		
		end
		if cfxZones.verbose then 
			trigger.action.outText("+++zne: accessing flag <" .. remainder .. ">", 30)
		end 
		rNum = cfxZones.getFlagValue(remainder, theZone)
	end 
	if rNum then 
		-- we have a comparison = ">", "=", "<" followed by a number 
		-- THEY TRIGGER EACH TIME lastVal <> currVal AND condition IS MET  
		if op == "=" then 
			return currVal == rNum and lastVal ~= currVal
		end
		
		if op == "#" or op == "~" then 
			return currVal ~= rNum and lastVal ~= currVal 
		end 
		
		if op == "<" then 
			return currVal < rNum and lastVal ~= currVal
		end
		
		if op == ">" then 
			return currVal > rNum and lastVal ~= currVal
		end
	end
	
	-- if we get here, we have an error 
	local zoneName = "<NIL>"
	if theZone then zoneName = theZone.name end 
	trigger.action.outText("+++Zne: illegal method constraints |" .. theMethod .. "| for zone " .. zoneName, 30 )
	return false 
end

-- WARNING: testZoneFlag must also support non-dmlZone!!!
function cfxZones.testZoneFlag(theZone, theFlagName, theMethod, latchName)
	-- returns two values: true/false method result, and curr value
	-- returns true if method constraints are met for flag theFlagName
	-- as defined by theMethod 
	if not theMethod then 
		theMethod = "change"
	end 
	
	-- will read and update theZone[latchName] as appropriate 
	if not theZone then 
		trigger.action.outText("+++Zne: no zone for testZoneFlag", 30)
		return nil, nil 
	end 
	if not theFlagName then 
		-- this is common, no error, only on verbose 
		if cfxZones.verbose then 
			trigger.action.outText("+++Zne: no flagName for zone " .. theZone.name .. " for testZoneFlag", 30)
		end 
		return nil, nil
	end
	if not latchName then 
		trigger.action.outText("+++Zne: no latchName for zone " .. theZone.name .. " for testZoneFlag", 30)
		return nil, nil 
	end
	-- get current value 
	local currVal = cfxZones.getFlagValue(theFlagName, theZone)
	
	-- get last value from latch
	local lastVal = theZone[latchName]
	if not lastVal then 
		trigger.action.outText("+++Zne: latch <" .. latchName .. "> not valid for zone " .. theZone.name, 30) -- intentional break here 
		return nil, nil
	end
	
	-- now, test by method 
	-- we should only test if currVal <> lastVal 
	if currVal == lastVal then
		return false, currVal
	end 
	
	local testResult = cfxZones.testFlagByMethodForZone(currVal, lastVal, theMethod, theZone)

	-- update latch by method
	theZone[latchName] = currVal 

	-- return result
	return testResult, currVal
end

function dmlZone:testZoneFlag(theFlagName, theMethod, latchName)
	local r, v = cfxZones.testZoneFlag(self, theFlagName, theMethod, latchName)
	return r, v 
end

function cfxZones.numberArrayFromString(inString, default) -- bridge
	return dcsCommon.numberArrayFromString(inString, default)
end
 

function cfxZones.flagArrayFromString(inString) -- dcsCommon bridge 
	return dcsCommon.flagArrayFromString(inString)
end


--
-- Drawing a Zone
--

function cfxZones.drawZone(theZone, lineColor, fillColor, markID)
	if not theZone then return 0 end 
	if not lineColor then lineColor = {0.8, 0.8, 0.8, 1.0} end
	if not fillColor then fillColor = {0.8, 0.8, 0.8, 0.2} end 
	if not markID then markID = dcsCommon.numberUUID() end 
	
	if theZone.isCircle then 
		trigger.action.circleToAll(-1, markID, theZone.point, theZone.radius, lineColor, fillColor, 1, true, "")
	else 
		local poly = theZone.poly
		trigger.action.quadToAll(-1, markID, poly[4], poly[3], poly[2], poly[1], lineColor, fillColor, 1, true, "") -- note: left winding to get fill color
	end
	
	return markID
end

function dmlZone:drawZone(lineColor, fillColor, markID)
	return cfxZones.drawZone(self, lineColor, fillColor, markID)
end

--
-- ===================
-- PROPERTY PROCESSING
-- =================== 
--

function cfxZones.getAllZoneProperties(theZone, caseInsensitive, numbersOnly) -- return as dict 
	if not caseInsensitive then caseInsensitive = false end 
	if not numbersOnly then numbersOnly = false end 
	if not theZone then return {} end 
	
	local dcsProps = theZone.properties -- zone properties in dcs format 
	local props = {}
	-- dcs has all properties as array with values .key and .value 
	-- so convert them into a dictionary 
	for i=1, #dcsProps do 
		local theProp = dcsProps[i]
		local theKey = "dummy"
		if string.len(theProp.key) > 0 then theKey = theProp.key end 
		if caseInsensitive then theKey = theKey:upper() end 
		local v = theProp.value 
		if numbersOnly then 
			v = tonumber(v)
			if not v then v = 0 end 
		end
		props[theKey] = v
	end
	return props 
end

function dmlZone:getAllZoneProperties(caseInsensitive, numbersOnly)
	return cfxZones.getAllZoneProperties(self, caseInsensitive, numbersOnly)
end

function cfxZones.extractPropertyFromDCS(theKey, theProperties)
-- trim
	theKey = dcsCommon.trim(theKey) 
--	make lower case conversion if not case sensitive
	if not cfxZones.caseSensitiveProperties then 
		theKey = string.lower(theKey)
	end

-- iterate all keys and compare to what we are looking for 	
	for i=1, #theProperties do
		local theP = theProperties[i]
		 
		local existingKey = dcsCommon.trim(theP.key)  
		if not cfxZones.caseSensitiveProperties then 
			existingKey = string.lower(existingKey)
		end
		if existingKey == theKey then 
			return theP.value
		end
		
		-- now check after removing all blanks 
		existingKey = dcsCommon.removeBlanks(existingKey)
		if existingKey == theKey then 
			return theP.value
		end
	end
	return nil 
end

function cfxZones.getZoneProperty(cZone, theKey)
	if not cZone then 
		trigger.action.outText("+++zone: no zone in getZoneProperty", 30)
		return nil
	end 
	if not theKey then 
		trigger.action.outText("+++zone: no property key in getZoneProperty for zone " .. cZone.name, 30)
		return 
	end	

	local props = cZone.properties
	local theVal = cfxZones.extractPropertyFromDCS(theKey, props)
	return theVal
end

function dmlZone:getZoneProperty(theKey)
	if not theKey then 
		trigger.action.outText("+++zone: no property key in OOP getZoneProperty for zone " .. self.name, 30)
		return nil  
	end	
	local props = self.properties
	local theVal = cfxZones.extractPropertyFromDCS(theKey, props)
	return theVal
end


function cfxZones.getStringFromZoneProperty(theZone, theProperty, default)
	if not default then default = "" end
-- OOP heavy duty test here
	local p = theZone:getZoneProperty(theProperty)
	if not p then return default end
	if type(p) == "string" then 
		p = dcsCommon.trim(p)
		if p == "" then p = default end 
		return p
	end
	return default -- warning. what if it was a number first?
end

function dmlZone:getStringFromZoneProperty(theProperty, default)
	if not default then default = "" end
	local p = self:getZoneProperty(theProperty)
	if not p then return default end
	if type(p) == "string" then 
		p = dcsCommon.trim(p)
		if p == "" then p = default end 
		return p
	end
	return default -- warning. what if it was a number first?
end

function cfxZones.getMinMaxFromZoneProperty(theZone, theProperty)
	local p = cfxZones.getZoneProperty(theZone, theProperty)
	local theNumbers = dcsCommon.splitString(p, " ")
	return tonumber(theNumbers[1]), tonumber(theNumbers[2])
end

function dmlZone:getMinMaxFromZoneProperty(theProperty)
	local p = self:getZoneProperty(theProperty)
	local theNumbers = dcsCommon.splitString(p, " ")
	return tonumber(theNumbers[1]), tonumber(theNumbers[2])
end

function cfxZones.randomInRange(minVal, maxVal) -- should be moved to dcsCommon
	if maxVal < minVal then 
		local t = minVal
		minVal = maxVal 
		maxVal = t
	end
	return cfxZones.randomDelayFromPositiveRange(minVal, maxVal)
end

function cfxZones.randomDelayFromPositiveRange(minVal, maxVal) -- should be moved to dcsCommon 
	if not maxVal then return minVal end 
	if not minVal then return maxVal end 
	local delay = maxVal
	if minVal > 0 and minVal < delay then 
		-- we want a randomized from time from minTime .. delay
		local varPart = delay - minVal + 1
		varPart = dcsCommon.smallRandom(varPart) - 1
		delay = minVal + varPart
	end
	return delay 
end

function cfxZones.getPositiveRangeFromZoneProperty(theZone, theProperty, default, defaultmax)
	-- reads property as string, and interprets as range 'a-b'. 
	-- if not a range but single number, returns both for upper and lower 
	--trigger.action.outText("***Zne: enter with <" .. theZone.name .. ">: range for property <" .. theProperty .. ">!", 30)
	if not default then default = 0 end 
	if not defaultmax then defaultmax = default end 
	
	local lowerBound = default
	local upperBound = defaultmax 
	
	local rangeString = cfxZones.getStringFromZoneProperty(theZone, theProperty, "")
	if dcsCommon.containsString(rangeString, "-") then 
		local theRange = dcsCommon.splitString(rangeString, "-")
		lowerBound = theRange[1]
		lowerBound = tonumber(lowerBound)
		upperBound = theRange[2]
		upperBound = tonumber(upperBound)
		if lowerBound and upperBound then
			-- swap if wrong order
			if lowerBound > upperBound then 
				local temp = upperBound
				upperBound = lowerBound
				lowerBound = temp 
			end

		else
			-- bounds illegal
			trigger.action.outText("+++Zne: illegal range  <" .. rangeString .. ">, using " .. default .. "-" .. defaultmax, 30)
			lowerBound = default
			upperBound = defaultmax 
		end
	else 
		upperBound = cfxZones.getNumberFromZoneProperty(theZone, theProperty, defaultmax) -- between pulses 
		lowerBound = upperBound
	end

	return lowerBound, upperBound
end

function dmlZone:getPositiveRangeFromZoneProperty(theProperty, default, defaultmax)
	local lo, up = cfxZones.getPositiveRangeFromZoneProperty(self, theProperty, default, defaultmax)
	return lo, up 
end


function cfxZones.hasProperty(theZone, theProperty) 
	if not theProperty then 
		trigger.action.outText("+++zne: WARNING - hasProperty called with nil theProperty for zone <" .. theZone.name .. ">", 30)
		return false 
	end 
	local foundIt = cfxZones.getZoneProperty(theZone, theProperty)
	if not foundIt then 
		-- check for possible forgotten or exchanged IO flags 
		if string.sub(theProperty, -1) == "?" then
			local lessOp = theProperty:sub(1,-2)
			if cfxZones.getZoneProperty(theZone, lessOp) ~= nil then 
				trigger.action.outText("*** NOTE: " .. theZone.name .. "'s property <" .. lessOp .. "> may be missing a Query ('?') symbol", 30)
			end
			local lessPlus = lessOp .. "!"
			if cfxZones.getZoneProperty(theZone, lessPlus) ~= nil then 
				trigger.action.outText("*** NOTE: " .. theZone.name .. "'s property <" .. lessOp .. "> may be using '!' instead of '?' for input", 30)
			end
			return false 
		end
		
		if string.sub(theProperty, -1) == "!" then 
			local lessOp = theProperty:sub(1,-2)
			if cfxZones.getZoneProperty(theZone, lessOp) ~= nil then 
				trigger.action.outText("*** NOTE: " .. theZone.name .. "'s property <" .. lessOp .. "> may be missing a Bang! ('!') symbol", 30)
			end
			local lessPlus = lessOp .. "?"
			if cfxZones.getZoneProperty(theZone, lessPlus) ~= nil then 
				trigger.action.outText("*** NOTE: " .. theZone.name .. "'s property <" .. lessOp .. "> may be using '!' instead of '?' for input", 30)
			end
			return false 
		end
		
		if string.sub(theProperty, -1) == ":" then 
			local lessOp = theProperty:sub(1,-2)
			if cfxZones.getZoneProperty(theZone, lessOp) ~= nil then 
				trigger.action.outText("*** NOTE: " .. theZone.name .. "'s property <" .. lessOp .. "> may be missing a colon (':') at end", 30)
			end
			return false 
		end
		
		if string.sub(theProperty, -1) == "#" then 
			local lessOp = theProperty:sub(1,-2)
			if cfxZones.getZoneProperty(theZone, lessOp) ~= nil then 
				trigger.action.outText("*** NOTE: " .. theZone.name .. "'s property <" .. lessOp .. "> may be missing a hash mark ('#') at end", 30)
			end
			return false 
		end
		
		return false 
	end
	return true 
end

function dmlZone:hasProperty(theProperty) 
	if not theProperty then 
		trigger.action.outText("+++zne: WARNING - hasProperty called with nil theProperty for zone <" .. self.name .. ">", 30)
		return false 
	end 
	local foundIt = self:getZoneProperty(theProperty)
	if not foundIt then 
		-- check for possible forgotten or exchanged IO flags 
		if string.sub(theProperty, -1) == "?" then
			local lessOp = theProperty:sub(1,-2)
			if self:getZoneProperty(lessOp) ~= nil then 
				trigger.action.outText("*** NOTE: " .. self.name .. "'s property <" .. lessOp .. "> may be missing a Query ('?') symbol", 30)
			end
			local lessPlus = lessOp .. "!"
			if self:getZoneProperty(lessPlus) ~= nil then 
				trigger.action.outText("*** NOTE: " .. self.name .. "'s property <" .. lessOp .. "> may be using '!' instead of '?' for input", 30)
			end
			return false 
		end
		
		if string.sub(theProperty, -1) == "!" then 
			local lessOp = theProperty:sub(1,-2)
			if self:getZoneProperty(lessOp) ~= nil then 
				trigger.action.outText("*** NOTE: " .. self.name .. "'s property <" .. lessOp .. "> may be missing a Bang! ('!') symbol", 30)
			end
			local lessPlus = lessOp .. "?"
			if self:getZoneProperty(lessPlus) ~= nil then 
				trigger.action.outText("*** NOTE: " .. self.name .. "'s property <" .. lessOp .. "> may be using '!' instead of '?' for input", 30)
			end
			return false 
		end
		
		if string.sub(theProperty, -1) == ":" then 
			local lessOp = theProperty:sub(1,-2)
			if self:getZoneProperty(lessOp) ~= nil then 
				trigger.action.outText("*** NOTE: " .. self.name .. "'s property <" .. lessOp .. "> may be missing a colon (':') at end", 30)
			end
			return false 
		end
		
		if string.sub(theProperty, -1) == "#" then 
			local lessOp = theProperty:sub(1,-2)
			if self:getZoneProperty(lessOp) ~= nil then 
				trigger.action.outText("*** NOTE: " .. self.name .. "'s property <" .. lessOp .. "> may be missing a hash mark ('#') at end", 30)
			end
			return false 
		end
		
		return false 
	end
	return true 
end

function cfxZones.getBoolFromZoneProperty(theZone, theProperty, defaultVal)
	if not defaultVal then defaultVal = false end 
	if type(defaultVal) ~= "boolean" then 
		defaultVal = false 
	end

	if not theZone then 
		trigger.action.outText("WARNING: NIL Zone in getBoolFromZoneProperty", 30)
		return defaultVal
	end


	local p = cfxZones.getZoneProperty(theZone, theProperty)
	if not p then return defaultVal end

	-- make sure we compare so default always works when 
	-- answer isn't exactly the opposite
	p = p:lower() 
	p = dcsCommon.trim(p) 
	if defaultVal == false then 
		-- only go true if exact match to yes or true 
		theBool = false 
		theBool = (p == 'true') or (p == 'yes') or (p == "1") or (p == 'on')
		return theBool
	end
	
	local theBool = true 
	-- only go false if exactly no or false or "0"
	theBool = (p ~= 'false') and (p ~= 'no') and (p ~= "0") and (p~="off")
	return theBool
end

function dmlZone:getBoolFromZoneProperty(theProperty, defaultVal)
	if not defaultVal then defaultVal = false end 
	if type(defaultVal) ~= "boolean" then 
		defaultVal = false 
	end

	local p = self:getZoneProperty(theProperty)
	if not p then return defaultVal end

	-- make sure we compare so default always works when 
	-- answer isn't exactly the opposite
	p = p:lower() 
	p = dcsCommon.trim(p) 
	if defaultVal == false then 
		-- only go true if exact match to yes or true 
		theBool = false 
		theBool = (p == 'true') or (p == 'yes') or (p == "1") or (p=="on")
		return theBool
	end
	
	local theBool = true 
	-- only go false if exactly no or false or "0"
	theBool = (p ~= 'false') and (p ~= 'no') and (p ~= "0") and (p ~= "off")
	return theBool
end

function cfxZones.getCoalitionFromZoneProperty(theZone, theProperty, default)
	if not default then default = 0 end
	local p = cfxZones.getZoneProperty(theZone, theProperty)
	if not p then return default end  
	if type(p) == "number" then -- can't currently really happen
		if p == 1 then return 1 end 
		if p == 2 then return 2 end 
		return 0
	end
	
	if type(p) == "string" then 
		if p == "1" then return 1 end 
		if p == "2" then return 2 end 
		if p == "0" then return 0 end 
		
		p = p:lower()
		
		if p == "red" then return 1 end 
		if p == "blue" then return 2 end 
		if p == "neutral" then return 0 end
		if p == "all" then return 0 end 
		return default 
	end
	
	return default 
end

function dmlZone:getCoalitionFromZoneProperty(theProperty, default)
	if not default then default = 0 end
	local p = self:getZoneProperty(theProperty)
	if not p then return default end  
	if type(p) == "number" then -- can't currently really happen
		if p == 1 then return 1 end 
		if p == 2 then return 2 end 
		return 0
	end
	
	if type(p) == "string" then 
		if p == "1" then return 1 end 
		if p == "2" then return 2 end 
		if p == "0" then return 0 end 
		
		p = p:lower()
		
		if p == "red" then return 1 end 
		if p == "blue" then return 2 end 
		if p == "neutral" then return 0 end
		if p == "all" then return 0 end 
		return default 
	end
	
	return default 
end

function cfxZones.getNumberFromZoneProperty(theZone, theProperty, default)
	if not default then default = 0 end
	default = tonumber(default)
	if not default then default = 0 end -- enforce default numbner as well 
	local p = cfxZones.getZoneProperty(theZone, theProperty)
	p = tonumber(p)
	if not p then p = default end 
	return p
end

function dmlZone:getNumberFromZoneProperty(theProperty, default) 
	if not default then default = 0 end
	default = tonumber(default)
	if not default then default = 0 end -- enforce default numbner as well 
	local p = self:getZoneProperty(theProperty)
	p = tonumber(p)
	if not p then p = default end 
	return p
end

function cfxZones.getVectorFromZoneProperty(theZone, theProperty, minDims, defaultVal)
	if not minDims then minDims = 0 end 
	if not defaultVal then defaultVal = 0 end 
	local s = cfxZones.getStringFromZoneProperty(theZone, theProperty, "")
	local sVec = dcsCommon.splitString(s, ",")
	local nVec = {}
	for idx, numString in pairs (sVec) do 
		local n = tonumber(numString)
		if not n then n = defaultVal end
		table.insert(nVec, n)
	end
	-- make sure vector contains at least minDims values 
	while #nVec < minDims do 
		table.insert(nVec, defaultVal)
	end
	
	return nVec 
end

function dmlZone:getVectorFromZoneProperty(theProperty, minDims, defaultVal)
	if not minDims then minDims = 0 end 
	if not defaultVal then defaultVal = 0 end 
	local s = self:getStringFromZoneProperty(theProperty, "")
	local sVec = dcsCommon.splitString(s, ",")
	local nVec = {}
	for idx, numString in pairs (sVec) do 
		local n = tonumber(numString)
		if not n then n = defaultVal end
		table.insert(nVec, n)
	end
	-- make sure vector contains at least minDims values 
	while #nVec < minDims do 
		table.insert(nVec, defaultVal)
	end
	
	return nVec 
end

function cfxZones.getRGBVectorFromZoneProperty(theZone, theProperty, defaultVal)
	if not defaultVal then defaultVal = {1.0, 1.0, 1.0} end 
	if #defaultVal ~=3 then defaultVal = {1.0, 1.0, 1.0} end
	local s = cfxZones.getStringFromZoneProperty(theZone, theProperty, "")
	local sVec = dcsCommon.splitString(s, ",")
	local nVec = {}
	for i = 1, 3 do 
		n = sVec[i]
		if n then n = tonumber(n) end 
		if not n then n = defaultVal[i] end 
		if n > 1.0 then n = 1.0 end
		if n < 0 then n = 0 end 
		nVec[i] = n
	end
	return nVec 
end

function dmlZone:getRGBVectorFromZoneProperty(theProperty, defaultVal)
	if not defaultVal then defaultVal = {1.0, 1.0, 1.0} end 
	if #defaultVal ~=3 then defaultVal = {1.0, 1.0, 1.0} end
	local s = self:getStringFromZoneProperty(theProperty, "")
	local sVec = dcsCommon.splitString(s, ",")
	local nVec = {}
	for i = 1, 3 do 
		n = sVec[i]
		if n then n = tonumber(n) end 
		if not n then n = defaultVal[i] end 
		if n > 1.0 then n = 1.0 end
		if n < 0 then n = 0 end 
		nVec[i] = n
	end
	return nVec 
end


function cfxZones.getRGBAVectorFromZoneProperty(theZone, theProperty, defaultVal)
	if not defaultVal then defaultVal = {1.0, 1.0, 1.0, 1.0} end 
	if #defaultVal ~=4 then defaultVal = {1.0, 1.0, 1.0, 1.0} end
	local s = cfxZones.getStringFromZoneProperty(theZone, theProperty, "")
	s = dcsCommon.trim(s)
	if s:sub(1,1) == "#" then 
		-- it's probably a "#RRGGBBAA" format hex string 
		local hVec = dcsCommon.hexString2RGBA(s)
		if hVec then return hVec end 
	end

	local sVec = dcsCommon.splitString(s, ",")
	local nVec = {}
	for i = 1, 4 do 
		n = sVec[i]
		if n then n = tonumber(n) end 
		if not n then n = defaultVal[i] end 
		if n > 1.0 then n = 1.0 end
		if n < 0 then n = 0 end 
		nVec[i] = n
	end
		
	return nVec 
end

function dmlZone:getRGBAVectorFromZoneProperty(theProperty, defaultVal)
	if not defaultVal then defaultVal = {1.0, 1.0, 1.0, 1.0} end 
	if #defaultVal ~=4 then defaultVal = {1.0, 1.0, 1.0, 1.0} end
	local s = self:getStringFromZoneProperty(theProperty, "")
	s = dcsCommon.trim(s)
	if s:sub(1,1) == "#" then 
		-- it's probably a "#RRGGBBAA" format hex string 
		local hVec = dcsCommon.hexString2RGBA(s)
		if hVec then return hVec end 
	end

	local sVec = dcsCommon.splitString(s, ",")
	local nVec = {}
	for i = 1, 4 do 
		n = sVec[i]
		if n then n = tonumber(n) end 
		if not n then n = defaultVal[i] end 
		if n > 1.0 then n = 1.0 end
		if n < 0 then n = 0 end 
		nVec[i] = n
	end
		
	return nVec 
end

function cfxZones.getRGBFromZoneProperty(theZone, theProperty, default)
	--if not default then default = {1.0, 1.0, 1.0} end -- white 
	local rawRGB = cfxZones.getVectorFromZoneProperty(theZone, theProperty, 3, 1.0)
	local retVal = {}
	for i = 1, 3 do 
		local cp = rawRGB[i]
		if cp > 1.0 then cp = 1.0 end
		if cp < 0 then cp = 0 end 
		retVal[i] = cp
	end
	return retVal
end

function dmlZone:getRGBFromZoneProperty(theProperty, default)
	--if not default then default = {1.0, 1.0, 1.0} end -- white 
	local rawRGB = self:getVectorFromZoneProperty(theProperty, 3, 1.0)
	local retVal = {}
	for i = 1, 3 do 
		local cp = rawRGB[i]
		if cp > 1.0 then cp = 1.0 end
		if cp < 0 then cp = 0 end 
		retVal[i] = cp
	end
	return retVal
end


function cfxZones.getSmokeColorStringFromZoneProperty(theZone, theProperty, default) -- smoke as 'red', 'green', or 1..5
	if not default then default = "red" end 
	local s = cfxZones.getStringFromZoneProperty(theZone, theProperty, default)
	s = s:lower()
	s = dcsCommon.trim(s)
	-- check numbers 
	if (s == "0") then return "green" end
	if (s == "1") then return "red" end
	if (s == "2") then return "white" end
	if (s == "3") then return "orange" end
	if (s == "4") then return "blue" end
	
	if s == "green" or
	   s == "red" or
	   s == "white" or
	   s == "orange" or
	   s == "blue" then return s end

	return default 
end

function dmlZone:getSmokeColorStringFromZoneProperty(theProperty, default) -- smoke as 'red', 'green', or 1..5
	if not default then default = "red" end 
	local s = self:getStringFromZoneProperty(theProperty, default)
	s = s:lower()
	s = dcsCommon.trim(s)
	-- check numbers 
	if (s == "0") then return "green" end
	if (s == "1") then return "red" end
	if (s == "2") then return "white" end
	if (s == "3") then return "orange" end
	if (s == "4") then return "blue" end
	
	if s == "green" or
	   s == "red" or
	   s == "white" or
	   s == "orange" or
	   s == "blue" then return s end

	return default 
end

function cfxZones.getFlareColorStringFromZoneProperty(theZone, theProperty, default) -- smoke as 'red', 'green', or 1..5
	if not default then default = "red" end 
	local s = cfxZones.getStringFromZoneProperty(theZone, theProperty, default)
	s = s:lower()
	s = dcsCommon.trim(s)
	-- check numbers 
	if (s == "rnd") then return "random" end 
	if (s == "0") then return "green" end
	if (s == "1") then return "red" end
	if (s == "2") then return "white" end
	if (s == "3") then return "yellow" end
	if (s == "-1") then return "random" end  
	
	if s == "green" or
	   s == "red" or
	   s == "white" or
	   s == "yellow" or 
	   s == "random" then
	return s end

	return default 
end

function dmlZone:getFlareColorStringFromZoneProperty(theProperty, default) -- smoke as 'red', 'green', or 1..5
	if not default then default = "red" end 
	local s = self:getStringFromZoneProperty(theProperty, default)
	s = s:lower()
	s = dcsCommon.trim(s)
	-- check numbers 
	if (s == "rnd") then return "random" end 
	if (s == "0") then return "green" end
	if (s == "1") then return "red" end
	if (s == "2") then return "white" end
	if (s == "3") then return "yellow" end
	if (s == "-1") then return "random" end  
	
	if s == "green" or
	   s == "red" or
	   s == "white" or
	   s == "yellow" or 
	   s == "random" then
	return s end

	return default 
end

--
-- Zone-based wildcard processing
-- 

-- process <z>
function cfxZones.processZoneStatics(inMsg, theZone)
	if theZone then 
		inMsg = inMsg:gsub("<z>", theZone.name)
	end
	return inMsg 
end

function dmlZone:processZoneStatics(inMsg, theZone)
	inMsg = inMsg:gsub("<z>", self.name)
	return inMsg 
end

-- process <t>, <lat>, <lon>, <ele>, <mgrs> 
function cfxZones.processSimpleZoneDynamics(inMsg, theZone, timeFormat, imperialUnits)
	if not inMsg then return "<nil inMsg>" end
	-- replace <t> with current mission time HMS
	local absSecs = timer.getAbsTime()-- + env.mission.start_time
	while absSecs > 86400 do 
		absSecs = absSecs - 86400 -- subtract out all days 
	end
	if not timeFormat then timeFormat = "<:h>:<:m>:<:s>" end 
	local timeString  = dcsCommon.processHMS(timeFormat, absSecs)
	local outMsg = inMsg:gsub("<t>", timeString)
	
	-- replace <lat> with lat of zone point and <lon> with lon of zone point 
	-- and <mgrs> with mgrs coords of zone point 
	local currPoint = cfxZones.getPoint(theZone)
	local lat, lon = coord.LOtoLL(currPoint)
	lat, lon = dcsCommon.latLon2Text(lat, lon)
	local alt = land.getHeight({x = currPoint.x, y = currPoint.z})
	if imperialUnits then 
		alt = math.floor(alt * 3.28084) -- feet 
	else 
		alt = math.floor(alt) -- meters 
	end 
	outMsg = outMsg:gsub("<lat>", lat)
	outMsg = outMsg:gsub("<lon>", lon)
	outMsg = outMsg:gsub("<ele>", alt)
	local grid = coord.LLtoMGRS(coord.LOtoLL(currPoint))
	local mgrs = grid.UTMZone .. ' ' .. grid.MGRSDigraph .. ' ' .. grid.Easting .. ' ' .. grid.Northing
	outMsg = outMsg:gsub("<mgrs>", mgrs)
	return outMsg
end 

-- process <v: flag>, <rsp: flag> <rrnd>
function cfxZones.processDynamicValues(inMsg, theZone, msgResponses)
	-- replace all occurences of <v: flagName> with their values 
	local pattern = "<v:%s*[%s%w%*%d%.%-_]+>" -- no list allowed but blanks and * and . and - and _ --> we fail on the other specials to keep this simple 
	local outMsg = inMsg
	repeat -- iterate all patterns one by one 
		local startLoc, endLoc = string.find(outMsg, pattern)
		if startLoc then 
			local theValParam = string.sub(outMsg, startLoc, endLoc)
			-- strip lead and trailer 
			local param = string.gsub(theValParam, "<v:%s*", "")
			param = string.gsub(param, ">","")
			-- param = dcsCommon.trim(param) -- trim is called anyway
			-- access flag
			local val = cfxZones.getFlagValue(param, theZone)
			val = tostring(val)
			if not val then val = "NULL" end 
			-- replace pattern in original with new val 
			outMsg = string.gsub(outMsg, pattern, val, 1) -- only one sub!
		end
	until not startLoc
	
	-- now process rsp 
	pattern = "<rsp:%s*[%s%w%*%d%.%-_]+>" -- no list allowed but blanks and * and . and - and _ --> we fail on the other specials to keep this simple 

	if msgResponses and (#msgResponses > 0) then -- only if this zone has an array
		--trigger.action.outText("enter response proccing", 30)
		repeat -- iterate all patterns one by one 
			local startLoc, endLoc = string.find(outMsg, pattern)
			if startLoc then 
				local theValParam = string.sub(outMsg, startLoc, endLoc)
				-- strip lead and trailer 
				local param = string.gsub(theValParam, "<rsp:%s*", "")
				param = string.gsub(param, ">","")
				
				-- access flag
				local val = cfxZones.getFlagValue(param, theZone)
				if not val or (val < 1) then val = 1 end 
				if val > msgResponses then val = msgResponses end 
				
				val = msgResponses[val]
				val = dcsCommon.trim(val)
				-- replace pattern in original with new val 
				outMsg = string.gsub(outMsg, pattern, val, 1) -- only one sub!
			end
		until not startLoc
		
		-- rnd response 
		local rndRsp = dcsCommon.pickRandom(msgResponses)
		outMsg = outMsg:gsub ("<rrnd>", rndRsp)
	end
	
	return outMsg
end

-- process <t: flag>
function cfxZones.processDynamicTime(inMsg, theZone, timeFormat)
	if not timeFormat then timeFormat = "<:h>:<:m>:<:s>" end
	-- replace all occurences of <t: flagName> with their values 
	local pattern = "<t:%s*[%s%w%*%d%.%-_]+>" -- no list allowed but blanks and * and . and - and _ --> we fail on the other specials to keep this simple 
	local outMsg = inMsg
	repeat -- iterate all patterns one by one 
		local startLoc, endLoc = string.find(outMsg, pattern)
		if startLoc then 
			local theValParam = string.sub(outMsg, startLoc, endLoc)
			-- strip lead and trailer 
			local param = string.gsub(theValParam, "<t:%s*", "")
			param = string.gsub(param, ">","")
			-- access flag
			local val = cfxZones.getFlagValue(param, theZone)
			-- use this to process as time value 
			--trigger.action.outText("time: accessing <" .. param .. "> and received <" .. val .. ">", 30)
			local timeString  = dcsCommon.processHMS(timeFormat, val)
			
			if not timeString then timeString = "NULL" end 
			-- replace pattern in original with new val 
			outMsg = string.gsub(outMsg, pattern, timeString, 1) -- only one sub!
		end
	until not startLoc
	return outMsg
end

-- process <lat/lon/ele/mgrs/lle/latlon/alt/vel/hdg/rhdg/type/player: zone/unit>
function cfxZones.processDynamicLoc(inMsg, imperialUnits, responses)
	local locales = {"lat", "lon", "ele", "mgrs", "lle", "latlon", "alt", "vel", "hdg", "rhdg", "type", "player"}
	local outMsg = inMsg
	local uHead = 0
	for idx, aLocale in pairs(locales) do 
		local pattern = "<" .. aLocale .. ":%s*[%s%w%*%d%.%-_]+>"
		repeat -- iterate all patterns one by one 
			local startLoc, endLoc = string.find(outMsg, pattern)
			if startLoc then
				local theValParam = string.sub(outMsg, startLoc, endLoc)
				-- strip lead and trailer 
				local param = string.gsub(theValParam, "<" .. aLocale .. ":%s*", "")
				param = string.gsub(param, ">","")
				-- find zone or unit
				param = dcsCommon.trim(param)
				local thePoint = nil 
				local tZone = cfxZones.getZoneByName(param)
				local tUnit = Unit.getByName(param)
				local spd = 0
				local angels = 0 
				local theType = "<errType>"
				local playerName = "Unknown"
				if tZone then
					theType = "Zone"
					playerName = "?zone?"
					thePoint = cfxZones.getPoint(tZone)
					if tZone.linkedUnit and Unit.isExist(tZone.linkedUnit) then 
						local lU = tZone.linkedUnit
						local masterPoint = lU:getPoint()
						thePoint.y = masterPoint.y 
						spd = dcsCommon.getUnitSpeed(lU)
						spd = math.floor(spd * 3.6)
						uHead = math.floor(dcsCommon.getUnitHeading(tUnit) * 57.2958) -- to degrees.
					else 
						-- since zones always have elevation of 0, 
						-- now get the elevation from the map 
						thePoint.y = land.getHeight({x = thePoint.x, y = thePoint.z})
					end
				elseif tUnit then 
					if Unit.isExist(tUnit) then
						theType = tUnit:getTypeName()
						if tUnit.getPlayerName and tUnit:getPlayerName() then
							playerName = tUnit:getPlayerName()
						end
						thePoint = tUnit:getPoint()
						spd = dcsCommon.getUnitSpeed(tUnit)
						-- convert m/s to km/h 
						spd = math.floor(spd * 3.6)
						uHead = math.floor(dcsCommon.getUnitHeading(tUnit) * 57.2958) -- to degrees. 
					end
				else 
					-- nothing to do, remove me.
				end

				local locString = "err"
				if thePoint then 
					-- now that we have a point, we can do locale-specific
					-- processing. return result in locString
					local lat, lon, alt = coord.LOtoLL(thePoint)
					lat, lon = dcsCommon.latLon2Text(lat, lon)
					angels = math.floor(thePoint.y) 
					if imperialUnits then 
						alt = math.floor(alt * 3.28084) -- feet
						spd = math.floor(spd * 0.539957) -- km/h to knots	
						angels = math.floor(angels * 3.28084)
					else 
						alt = math.floor(alt) -- meters 
					end 
					
					if angels > 1000 then 
						angels = math.floor(angels / 100) * 100 
					end
					
					if aLocale == "lat" then locString = lat 
					elseif aLocale == "lon" then locString = lon 
					elseif aLocale == "ele" then locString = tostring(alt)
					elseif aLocale == "lle" then locString = lat .. " " .. lon .. " ele " .. tostring(alt) 
					elseif aLocale == "latlon" then locString = lat .. " " .. lon 
					elseif aLocale == "alt" then locString = tostring(angels) -- don't confuse alt and angels, bad var naming here
					elseif aLocale == "vel" then locString = tostring(spd)
					elseif aLocale == "hdg" then locString = tostring(uHead)
					elseif aLocale == "type" then locString = theType 
					elseif aLocale == "player" then locString = playerName 
					elseif aLocale == "rhdg" and (responses) then 
						local offset = cfxZones.rspMapper360(uHead, #responses)
						locString = dcsCommon.trim(responses[offset])
					else 
						-- we have mgrs
						local grid = coord.LLtoMGRS(coord.LOtoLL(thePoint))
						locString = grid.UTMZone .. ' ' .. grid.MGRSDigraph .. ' ' .. grid.Easting .. ' ' .. grid.Northing
					end
				end
				-- replace pattern in original with new val 
				outMsg = string.gsub(outMsg, pattern, locString, 1) -- only one sub!
			end -- if startloc
		until not startLoc
	end -- for all locales 
	return outMsg
end

-- process reference that can be flag, Zone, or unit.
-- i.e. <coa: xyz>
function cfxZones.processDynamicVZU(inMsg)
local locales = {"coa",}
	local outMsg = inMsg
	local uHead = 0
	for idx, aLocale in pairs(locales) do 
		local pattern = "<" .. aLocale .. ":%s*[%s%w%*%d%.%-_]+>" -- e.g. "<coa: flag Name>
		repeat -- iterate all patterns one by one 
			local startLoc, endLoc = string.find(outMsg, pattern)
			if startLoc then
				local theValParam = string.sub(outMsg, startLoc, endLoc)
				-- strip lead and trailer 
				local param = string.gsub(theValParam, "<" .. aLocale .. ":%s*", "") -- remove "<coa:"
				param = string.gsub(param, ">","") -- remove trailing ">"
				-- find zone or unit
				param = dcsCommon.trim(param) -- param = "flag Name"
				local tZone = cfxZones.getZoneByName(param)
				local tUnit = Unit.getByName(param)

				local locString = "err"
				if aLocale == "coa" then
					coa = trigger.misc.getUserFlag(param)
					if tZone then coa = tZone.owner end 
					if tUnit and Unit:isExist(tUnit) then coa = tUnit:getCoalition() end 
					locString = dcsCommon.coalition2Text(coa)
				end

				outMsg = string.gsub(outMsg, pattern, locString, 1) -- only one sub!
			end -- if startloc
		until not startLoc
	end -- for all locales 
	return outMsg
end

-- process two-value vars that can be flag or unit and return interpreted value
-- i.e. <alive: Aerial-1-1>
function cfxZones.processDynamicValueVU(inMsg)
local locales = {"yes", "true", "alive", "in"}
	local outMsg = inMsg
	local uHead = 0
	for idx, aLocale in pairs(locales) do 
		local pattern = "<" .. aLocale .. ":%s*[%s%w%*%d%.%-_]+>" -- e.g. "<yes: flagOrUnitName>
		repeat -- iterate all patterns one by one 
			local startLoc, endLoc = string.find(outMsg, pattern)
			if startLoc then
				local theValParam = string.sub(outMsg, startLoc, endLoc)
				-- strip lead and trailer 
				local param = string.gsub(theValParam, "<" .. aLocale .. ":%s*", "") -- remove "<alive:"
				param = string.gsub(param, ">","") -- remove trailing ">"
				-- find zone or unit
				param = dcsCommon.trim(param) -- param = "flagOrUnitName"
				local tUnit = Unit.getByName(param)
				local yesNo = trigger.misc.getUserFlag(param) ~= 0
				if tUnit then yesNo = Unit.isExist(tUnit) end
				local locString = "err"
				if aLocale == "yes" then					
					if yesNo then locString = "yes" else locString = "no" end
				elseif aLocale == "true" then 
					if yesNo then locString = "true" else locString = "false" end 
				elseif aLocale == "alive" then 
					if yesNo then locString = "alive" else locString = "dead" end
				elseif aLocale == "in" then 
					if yesNo then locString = "in" else locString = "out" end
				end

				outMsg = string.gsub(outMsg, pattern, locString, 1) -- only one sub!
			end -- if startloc
		until not startLoc
	end -- for all locales 
	return outMsg
end

function cfxZones.processDynamicAB(inMsg, locale)
	local outMsg = inMsg
	if not locale then locale = "A/B" end 
	
	-- <A/B: flagOrUnitName [val A | val B]>
	local replacerValPattern = "<".. locale .. ":%s*[%s%w%*%d%.%-_]+" .. "%[[%s%w]+|[%s%w]+%]"..">"
	repeat 
		local startLoc, endLoc = string.find(outMsg, replacerValPattern)
		if startLoc then 
			local rp = string.sub(outMsg, startLoc, endLoc)
			-- get val/unit name 
			local valA, valB = string.find(rp, ":%s*[%s%w%*%d%.%-_]+%[")
			local val = string.sub(rp, valA+1, valB-1)
			val = dcsCommon.trim(val)
			-- get left and right 
			local leftA, leftB = string.find(rp, "%[[%s%w]+|" ) -- from "[" to "|"
			local rightA, rightB = string.find(rp, "|[%s%w]+%]") -- from "|" to "]"
			left = string.sub(rp, leftA+1, leftB-1)
			left = dcsCommon.trim(left)
			right = string.sub(rp, rightA+1, rightB-1)
			right = dcsCommon.trim(right)		
			local yesno = false
			-- see if unit exists
			local theUnit = Unit.getByName(val)
			if theUnit then 
				yesno = Unit:isExist(theUnit)
			else 
				yesno = trigger.misc.getUserFlag(val) ~= 0
			end

			local locString = left 
			if yesno then locString = right end 
			outMsg = string.gsub(outMsg, replacerValPattern, locString, 1)
		end
	until not startLoc 
	return outMsg
end

function cfxZones.rspMapper360(directionInDegrees, numResponses)
	-- maps responses around a clock. Clock has 12 'responses' (12, 1, .., 11), 
	-- with the first (12) also mapping to the last half arc 
	-- this method dynamically 'winds' the responses around 
	-- a clock and returns the index of the message to display 
	if numResponses < 1 then numResponses = 1 end 
	directionInDegrees = math.floor(directionInDegrees) 
	while directionInDegrees < 0 do directionInDegrees = directionInDegrees + 360 end 
	while directionInDegrees >= 360 do directionInDegrees = directionInDegrees - 360 end 
	-- now we have 0..360 
	-- calculate arc per item 
	local arcPerItem = 360 / numResponses
	local halfArc = arcPerItem / 2

	-- we now map 0..360 to (0-halfArc..360-halfArc) by shifting 
	-- direction by half-arc and clipping back 0..360
	-- and now we can directly derive the index of the response 
	directionInDegrees = directionInDegrees + halfArc
	if directionInDegrees >= 360 then directionInDegrees = directionInDegrees - 360 end 
	
	local index = math.floor(directionInDegrees / arcPerItem) + 1 -- 1 .. numResponses 
	
	return index 
end

-- replaces dcsCommon with same name 
-- timeFormat is optional, default is "<:h>:<:m>:<:s>"
-- imperialUnits is optional, defaults to meters 
-- responses is an array of string, defaults to {}
function cfxZones.processStringWildcards(inMsg, theZone, timeFormat, imperialUnits, responses)
	if not inMsg then return "<nil inMsg>" end
	local formerType = type(inMsg)
	if formerType ~= "string" then inMsg = tostring(inMsg) end
	if not inMsg then inMsg = "<inMsg is incompatible type " .. formerType .. ">" end
	local theMsg = inMsg
	-- process common DCS stuff like /n 
	theMsg = dcsCommon.processStringWildcards(theMsg) -- call old inherited
	-- process <z>
	theMsg = cfxZones.processZoneStatics(theMsg, theZone)
	-- process <t>, <lat>, <lon>, <ele>, <mgrs>
	theMsg = cfxZones.processSimpleZoneDynamics(theMsg, theZone, timeFormat, imperialUnits)
	-- process <v: flag>, <rsp: flag> <rrnd>
	theMsg = cfxZones.processDynamicValues(theMsg, theZone, responses)
	-- process <t: flag>
	theMsg = cfxZones.processDynamicTime(theMsg, theZone, timeFormat)
	-- process <lat/lon/ele/mgrs/lle/latlon/alt/vel/hdg/rhdg/type/player: zone/unit>
	theMsg = cfxZones.processDynamicLoc(theMsg, imperialUnits, responses)
    -- process values that can be derived from flag (default), zone or unit 
	theMsg = cfxZones.processDynamicVZU(theMsg)
	theMsg = cfxZones.processDynamicAB(theMsg)
	theMsg = cfxZones.processDynamicValueVU(theMsg)
	return theMsg
end

--
-- ============
-- MOVING ZONES 
-- ============ 
-- 
-- Moving zones contain a link to their unit
-- they are always located at an offset (x,z) or delta, phi 
-- to their master unit. delta phi allows adjustment for heading
-- The cool thing about moving zones in cfx is that they do not
-- require special handling, they are always updated 
-- and work with 'pointinzone' etc automatically

-- Always works on cfx Zones, NEVER on DCS zones.
--
-- requires that readFromDCS has been done
--
function cfxZones.getDCSOrigin(aZone)
	local o = {}
	o.x = aZone.dcsOrigin.x
	o.y = 0
	o.z = aZone.dcsOrigin.z 
	return o
end

function dmlZone:getDCSOrigin()
	local o = {}
	if not self.dcsOrigin then 
		trigger.action.outText("dmlZone (OOP): no dcsOrigin defined for zone <" .. self.name .. ">", 30)
		o.x = 0
		o.y = 0
		o.z = 0
	else
		o.x = self.dcsOrigin.x
		o.y = 0
		o.z = self.dcsOrigin.z 
	end 
	return o
end

function cfxZones.getLinkedUnit(theZone)
	if not theZone then return nil end 
	if not theZone.linkedUnit then return nil end 
	if not Unit.isExist(theZone.linkedUnit) then return nil end 
	return theZone.linkedUnit 
end

function dmlZone:getLinkedUnit()
	if not self.linkedUnit then return nil end 
	if not Unit.isExist(self.linkedUnit) then return nil end 
	return self.linkedUnit 
end

function cfxZones.getPoint(aZone, getHeight) -- always works, even linked, returned point can be reused
-- returned y (when using getHeight) is that of the land, else 0 
	if not getHeight then getHeight = false end 
	if aZone.linkedUnit then 
		local theUnit = aZone.linkedUnit
		-- has a link. is link existing?
		if Unit.isExist(theUnit) then 
			-- updates zone position 
			cfxZones.centerZoneOnUnit(aZone, theUnit)
			local dx = aZone.dx
			local dy = aZone.dy
			if aZone.useHeading then 
				dx, dy = cfxZones.calcHeadingOffset(aZone, theUnit)
			end
			cfxZones.offsetZone(aZone, dx, dy)
		end
	end
	local thePos = {}
	thePos.x = aZone.point.x
	thePos.z = aZone.point.z
	if not getHeight then 
		thePos.y = 0 -- aZone.y 
	else 
		thePos.y = land.getHeight({x = thePos.x, y = thePos.z})
	end
	
	return thePos 
end

function dmlZone:getPoint(getHeight)
	if not getHeight then getHeight = false end 
	if self.linkedUnit then 
		local theUnit = self.linkedUnit
		-- has a link. is link existing?
		if Unit.isExist(theUnit) then 
			-- updates zone position 
			self:centerZoneOnUnit(theUnit)
			local dx = self.dx
			local dy = self.dy
			if self.useHeading then 
				dx, dy = self:calcHeadingOffset(theUnit)
			end
			self:offsetZone(dx, dy)
		end
	end
	local thePos = {}
	thePos.x = self.point.x
	thePos.z = self.point.z
	if not getHeight then 
		thePos.y = 0 -- aZone.y 
	else 
		thePos.y = land.getHeight({x = thePos.x, y = thePos.z})
	end
	
	return thePos 
end

function dmlZone:getName() -- no cfxZones.bridge!
	return self.name 
end

function cfxZones.linkUnitToZone(theUnit, theZone, dx, dy) -- note: dy is really Z, don't get confused!!!!
	theZone.linkedUnit = theUnit
	if not dx then dx = 0 end
	if not dy then dy = 0 end 
	theZone.dx = dx
	theZone.dy = dy 
	theZone.rxy = math.sqrt(dx * dx + dy * dy) -- radius 
	local unitHeading = dcsCommon.getUnitHeading(theUnit)
	local bearingOffset = math.atan2(dy, dx) -- rads 
	if bearingOffset < 0 then bearingOffset = bearingOffset + 2 * 3.141592 end 

	local dPhi = bearingOffset - unitHeading
	if dPhi < 0 then dPhi = dPhi + 2 * 3.141592 end
	if (theZone.verbose and theZone.useHeading) then 
		trigger.action.outText("Zone is at <" .. math.floor(57.2958 * dPhi) .. "> relative to unit heading", 30)
	end
	theZone.dPhi = dPhi -- constant delta between unit heading and 
	-- direction to zone 
	theZone.uHdg = unitHeading -- original unit heading to turn other 
	-- units if need be 
end

function dmlZone:linkUnitToZone(theUnit, dx, dy) -- note: dy is really Z, don't get confused!!!!
	self.linkedUnit = theUnit
	if not dx then dx = 0 end
	if not dy then dy = 0 end 
	self.dx = dx
	self.dy = dy 
	self.rxy = math.sqrt(dx * dx + dy * dy) -- radius 
	local unitHeading = dcsCommon.getUnitHeading(theUnit)
	local bearingOffset = math.atan2(dy, dx) -- rads 
	if bearingOffset < 0 then bearingOffset = bearingOffset + 2 * 3.141592 end 

	local dPhi = bearingOffset - unitHeading
	if dPhi < 0 then dPhi = dPhi + 2 * 3.141592 end
	if (self.verbose and self.useHeading) then 
		trigger.action.outText("Zone <" .. self.name .. "> is at <" .. math.floor(57.2958 * dPhi) .. "> relative to unit heading", 30)
	end
	self.dPhi = dPhi -- constant delta between unit heading and 
	-- direction to zone 
	self.uHdg = unitHeading -- original unit heading to turn other 
	-- units if need be 
end

function cfxZones.zonesLinkedToUnit(theUnit) -- returns all zones linked to this unit 
	if not theUnit then return {} end 
	local linkedZones = {}
	for idx, theZone in pairs (cfxZones.zones) do 
		if theZone.linkedUnit == theUnit then 
			table.insert(linkedZones, theZone)
		end
	end
	return linkedZones
end

function cfxZones.calcHeadingOffset(aZone, theUnit)
	-- recalc dx and dy based on ry and current heading 
	-- since 0 degrees is [0,1] = [0,r] the calculation of 
	-- rotated coords can be simplified from 
	-- xr = x cos phi - y sin phi = -r sin phi
	-- yr = y cos phi + x sin phi = r cos phi 
	local unitHeading = dcsCommon.getUnitHeading(theUnit)
	-- add heading offset 
	local zoneBearing = unitHeading + aZone.dPhi 
	if zoneBearing > 2 * 3.141592 then zoneBearing = zoneBearing - 2 * 3.141592 end 
					
	-- in DCS, positive x is north (wtf?) and positive z is east 
	local dy = (-aZone.rxy) * math.sin(zoneBearing)
	local dx = aZone.rxy * math.cos(zoneBearing)
	return dx, -dy -- note: dy is z coord!!!!
end

function dmlZone:calcHeadingOffset(theUnit)
	local unitHeading = dcsCommon.getUnitHeading(theUnit)
	local zoneBearing = unitHeading + self.dPhi 
	if zoneBearing > 2 * 3.141592 then zoneBearing = zoneBearing - 2 * 3.141592 end 
	-- in DCS, positive x is north (wtf?) and positive z is east 
	local dy = (-self.rxy) * math.sin(zoneBearing)
	local dx = self.rxy * math.cos(zoneBearing)
	return dx, -dy -- note: dy is z coord!!!!
end


function cfxZones.updateMovingZones()
	cfxZones.updateSchedule = timer.scheduleFunction(cfxZones.updateMovingZones, {}, timer.getTime() + 1/cfxZones.ups)
	-- simply scan all cfx zones for the linkName property, and if present
	-- update the zone's points
	for aName,aZone in pairs(cfxZones.zones) do
		-- only do this if ther is a linkName property, 
		-- else this zone isn't linked. link name is harmonized from 
        -- both linkUnit non-DML and linedUnit DML		
		if aZone.linkName then 
			if aZone.linkBroken then 
				-- try to relink 
				cfxZones.initLink(aZone)
			else --if aZone.linkName then  
				-- always re-acquire linkedUnit via Unit.getByName()
				-- this way we gloss over any replacements via spawns
				aZone.linkedUnit = Unit.getByName(aZone.linkName)
			end
			
			if aZone.linkedUnit then 
				local theUnit = aZone.linkedUnit
				-- has a link. is link existing?
				if theUnit:isExist() then 
					cfxZones.centerZoneOnUnit(aZone, theUnit)
					local dx = aZone.dx 
					local dy = aZone.dy -- this is actually z 
					if aZone.useHeading then 
						dx, dy = cfxZones.calcHeadingOffset(aZone, theUnit)
					end
					cfxZones.offsetZone(aZone, dx, dy)
				else 
					-- we lost link (track level)
					aZone.linkBroken = true 
					aZone.linkedUnit = nil 
				end
			else 
				-- we lost link (top level)
				aZone.linkBroken = true 
				aZone.linkedUnit = nil 
			end
		else 
			-- this zone isn't linked
		end
	end
end

function cfxZones.initLink(theZone)
	theZone.linkBroken = true 
	theZone.linkedUnit = nil 
	theUnit = Unit.getByName(theZone.linkName)
	if theUnit then

		local dx = 0
		local dz = 0
		if theZone.useOffset or theZone.useHeading then 
			local A = cfxZones.getDCSOrigin(theZone)
			local B = theUnit:getPoint()
			local delta = dcsCommon.vSub(A,B) 
			dx = delta.x 
			dz = delta.z
		end
		cfxZones.linkUnitToZone(theUnit, theZone, dx, dz) -- also sets theZone.linkedUnit

		if theZone.verbose then 
			trigger.action.outText("Link established for zone <" .. theZone.name .. "> to unit <" .. theZone.linkName .. ">: dx=<" .. math.floor(dx) .. ">, dz=<" .. math.floor(dz) .. "> dist = <" .. math.floor(math.sqrt(dx * dx + dz * dz)) .. ">" , 30)
		end 
		theZone.linkBroken = nil 

	else 
		if theZone.verbose then 
			trigger.action.outText("Linked unit: no unit <" .. theZone.linkName .. "> to link <" .. theZone.name .. "> to", 30)
		end
	end
end

function dmlZone:initLink()
	self.linkBroken = true 
	self.linkedUnit = nil 
	theUnit = Unit.getByName(self.linkName)
	if theUnit then

		local dx = 0
		local dz = 0
		if self.useOffset or self.useHeading then 
			local A = self:getDCSOrigin()
			local B = theUnit:getPoint()
			local delta = dcsCommon.vSub(A,B) 
			dx = delta.x 
			dz = delta.z
		end
		self:linkUnitToZone(theUnit, dx, dz) -- also sets theZone.linkedUnit

		if self.verbose then 
			trigger.action.outText("Link established for zone <" .. self.name .. "> to unit <" .. self.linkName .. ">: dx=<" .. math.floor(dx) .. ">, dz=<" .. math.floor(dz) .. "> dist = <" .. math.floor(math.sqrt(dx * dx + dz * dz)) .. ">" , 30)
		end 
		self.linkBroken = nil 

	else 
		if self.verbose then 
			trigger.action.outText("Linked unit: no unit <" .. self.linkName .. "> to link <" .. self.name .. "> to", 30)
		end
	end
end

function cfxZones.startMovingZones()
	-- read all zones, and look for a property called 'linkedUnit'
	-- which will make them a linked zone if there is a unit that exists
	-- also suppors 'useOffset' and 'useHeading'
	for aName,aZone in pairs(cfxZones.zones) do
		
		local lU = nil 
		-- check if DCS zone has the linkUnit new attribute introduced in 
		-- late 2022 with 2.8
		if aZone.dcsZone.linkUnit then 
			local theID = aZone.dcsZone.linkUnit 
			lU = dcsCommon.getUnitNameByID(theID)
			if not lU then 
				trigger.action.outText("WARNING: Zone <" .. aZone.name .. ">: cannot resolve linked unit ID <" .. theID .. ">", 30)
				lU = "***DML link err***"
			end
		elseif cfxZones.hasProperty(aZone, "linkedUnit") then 
			lU = cfxZones.getZoneProperty(aZone, "linkedUnit")
		end
		
		-- sanity check 
		if aZone.dcsZone.linkUnit and cfxZones.hasProperty(aZone, "linkedUnit") then 
			trigger.action.outText("WARNING: Zone <" .. aZone.name .. "> has dual unit link definition. Will use link to unit <" .. lU .. ">", 30)
		end
		
		if lU then 
			aZone.linkName = lU
			aZone.useOffset = cfxZones.getBoolFromZoneProperty(aZone, "useOffset", false)
			aZone.useHeading = cfxZones.getBoolFromZoneProperty(aZone, "useHeading", false)
			
			cfxZones.initLink(aZone)

		end
		
	end
end

--
-- marking zones 
--

function cfxZones.spreadNObjectsOverLine(theZone, n, objType, left, right, cty) -- leaves last position free 
	trigger.action.outText("left = " .. dcsCommon.point2text(left) .. ", right = " .. dcsCommon.point2text(right),30)
	
	local a = {x=left.x, y=left.z}
	local b = {x=right.x, y=right.z}
	local dir = dcsCommon.vSub(b,a) -- vector from left to right
	local dirInc = dcsCommon.vMultScalar(dir, 1/n) 
	local count = 0 
	local p = {x=left.x, y = left.z}
	local baseName = dcsCommon.uuid(theZone.name)
	while count < n do 
		local theStaticData = dcsCommon.createStaticObjectData(dcsCommon.uuid(theZone.name), objType)
		dcsCommon.moveStaticDataTo(theStaticData, p.x, p.y)
		local theObject = coalition.addStaticObject(cty, theStaticData)
		p = dcsCommon.vAdd(p, dirInc) 
		count = count + 1
	end
end

function cfxZones.markZoneWithObjects(theZone, objType, qtrNum, markCenter, cty) -- returns set 
	if not objType then objType = "Black_Tyre_RF" end 
	if not qtrNum then qtrNum = 3 end -- +1 for number of marks per quarter 
	if not cty then cty = dcsCommon.getACountryForCoalition(0) end -- some neutral county
	local p = theZone:getPoint()
	local newObjects = {}
	
	if theZone.isPoly then 
		-- we place 4 * (qtrnum + 1) objects around the edge of the zone 
		-- we mark each poly along v-->v+1, placing ip and qtrNum additional points 
		local o = cfxZones.spreadNObjectsOverLine(theZone, qtrNum + 1, objType, theZone.poly[1], theZone.poly[2], cty)
		local p = cfxZones.spreadNObjectsOverLine(theZone, qtrNum + 1, objType, theZone.poly[2], theZone.poly[3], cty)
		local q = cfxZones.spreadNObjectsOverLine(theZone, qtrNum + 1, objType, theZone.poly[3], theZone.poly[4], cty)
		local r = cfxZones.spreadNObjectsOverLine(theZone, qtrNum + 1, objType, theZone.poly[4], theZone.poly[1], cty)
		o = dcsCommon.combineTables(o,p)
		p = dcsCommon.combineTables(q,r)
		newObjects = dcsCommon.combineTables(o,p)
		
	else 
		local numObjects = (qtrNum + 1) * 4
		local degrees = 3.14157 / 180
		local degreeIncrement = (360 / numObjects) * degrees
		local currDegree = 0
		local radius = theZone.radius
		for i=1, numObjects do 
			local ox = p.x + math.cos(currDegree) * radius
			local oy = p.z + math.sin(currDegree) * radius -- note: z!
			local theStaticData = dcsCommon.createStaticObjectData(dcsCommon.uuid(theZone.name), objType)
			dcsCommon.moveStaticDataTo(theStaticData, ox, oy)
			local theObject = coalition.addStaticObject(cty, theStaticData)
			table.insert(newObjects, theObject)
			currDegree = currDegree + degreeIncrement
		end
	end
	
	if markCenter then 
		-- also mark the center 
		local theObject = cfxZones.markPointWithObject(p, objType, cty)
		table.insert(newObjects, theObject)
	end 	
	
	return newObjects
end

function dmlZone:markZoneWithObjects(objType, qtrNum, markCenter, cty) -- returns set 
	return cfxZones.markZoneWithObjects(self, objType, qtrNum, markCenter)
end

function cfxZones.markCenterWithObject(theZone, objType, cty) -- returns object
	local p = cfxZones.getPoint(theZone)
	local theObject = cfxZones.markPointWithObject(theZone, p, objType, cty)
	return theObject
end

function dmlZone:markCenterWithObject(objType, cty) -- returns object 
	return cfxZones.markCenterWithObject(self, objType, cty)
end

function cfxZones.markPointWithObject(theZone, p, theType, cty) -- returns object 
	if not cty then cty = dcsCommon.getACountryForCoalition(0) end
	local ox = p.x
	local oy = p.y 	
	if p.z then oy = p.z end -- support vec 2 and vec 3 
	local theStaticData = dcsCommon.createStaticObjectData(dcsCommon.uuid(theZone.name), theType)
	dcsCommon.moveStaticDataTo(theStaticData, ox, oy)
	local theObject = coalition.addStaticObject(cty, theStaticData)
	return theObject
end

function dmlZone:markPointWithObject(p, theType, cty) -- returns object 
	return cfxZones.markPointWithObject(self, p, theType, cty)
end
--
-- ===========
-- INIT MODULE
-- ===========
--

function cfxZones.initZoneVerbosity()
	for aName,aZone in pairs(cfxZones.zones) do
		-- support for zone-local verbose flag 
		aZone.verbose = cfxZones.getBoolFromZoneProperty(aZone, "verbose", false)
	end
end

function cfxZones.init()
	-- read all zones into my own db
	cfxZones.readFromDCS(true) -- true: erase old

	-- pre-read zone owner for all zones
	-- much like verbose, all zones have owner
    for n, aZone in pairs(cfxZones.zones) do
		aZone.owner = cfxZones.getCoalitionFromZoneProperty(aZone, "owner", 0)
	end
		
	-- enable all zone's verbose flags if present
	-- must be done BEFORE we start the moving zones 
	cfxZones.initZoneVerbosity()
	
	-- now initialize moving zones
	cfxZones.startMovingZones()
	cfxZones.updateMovingZones() -- will auto-repeat
	
	trigger.action.outText("cf/x Zones v".. cfxZones.version .. ": loaded, zones:" .. dcsCommon.getSizeOfTable(cfxZones.zones), 30)

end

-- get everything rolling
cfxZones.init()

persistence = {}
persistence.version = "2.0.0"
persistence.ups = 1 -- once every 1 seconds 
persistence.verbose = false 
persistence.active = false 
persistence.saveFileName = nil -- "mission data.txt"
persistence.sharedDir = nil -- not yet implemented
persistence.missionDir = nil -- set at start 
persistence.saveDir = nil -- set at start 
persistence.name = "persistence" -- for cfxZones 
persistence.missionData = {} -- loaded from file 
persistence.requiredLibs = {
	"dcsCommon", 
	"cfxZones",  
}
--[[--
	Version History 
	2.0.0 - dml zones, OOP
			cleanup
	
	PROVIDES LOAD/SAVE ABILITY TO MODULES
	PROVIDES STANDALONE/HOSTED SERVER COMPATIBILITY
--]]--

-- in order to work, HOST MUST DESANITIZE lfs and io 

--
-- flags to save. can be added to by saveFlags attribute 
--
persistence.flagsToSave = {} -- simple table 
persistence.callbacks = {} -- cbblocks, dictionary by name


--
-- modules register here
--
function persistence.registerModule(name, callbacks)
	-- callbacks is a table with the following entries
	-- callbacks.persistData - method that returns a table
	-- note that name is also what the data is saved under
	-- and must be the one given when you retrieve it later
	persistence.callbacks[name] = callbacks
	if persistence.verbose then 
		trigger.action.outText("+++persistence: module <" .. name .. "> registered itself", 30)
	end
end

function persistence.registerFlagsToSave(flagNames, theZone)
	-- name can be single flag name or anything that 
	-- a zone definition has to offer, including local 
	-- flags. 
    -- flags can be passed like this: "a, 4-19, 99, kills, *lcl"
	-- if you pass a local flag, you must pass the zone 
	-- or "persisTEMP" will be used 
	
	if not theZone then theZone = cfxZones.createSimpleZone("persisTEMP") end 
	local newFlags = dcsCommon.flagArrayFromString(flagNames, persistence.verbose)
	
	-- mow process all new flags and add them to the list of flags 
	-- to save 
	for idx, flagName in pairs(newFlags) do 
		if dcsCommon.stringStartsWith(flagName, "*") then 
			flagName = theZone.name .. flagName 
		end
		table.insert(persistence.flagsToSave, flagName)
	end
end
--
-- registered modules call this to get their data 
--
function persistence.getSavedDataForModule(name)
	if not persistence.active then return nil end 
	if not persistence.hasData then return nil end 
	if not persistence.missionData then return end 
	
	return persistence.missionData[name] -- simply get the modules data block
end


--
-- Shared Data API 
--
function persistence.getSharedDataFor(name, item) -- not yet finalized
end

function persistence.putSharedDataFor(data, name, item) -- not yet finalized
end

--
-- helper meths
--

function persistence.hasFile(path) --check if file exists at path
-- will also return true for a directory, follow up with isDir 

    local attr = lfs.attributes(path) 
	if attr then
		return true, attr.mode
	end 
	
	if persistence.verbose then 
		trigger.action.outText("isFile: attributes not found for <" .. path .. ">", 30)
	end 
	
	return false, "<err>"
end

function persistence.isDir(path) -- check if path is a directory
	local success, mode = persistence.hasFile(path)
	if success then 
		success = (mode == "directory")
	end
	return success
end


--
-- Main save meths
--
function persistence.saveText(theString, fileName, shared, append)
	if not persistence.active then return false end 
	if not fileName then 
		trigger.action.outText("+++persistence: saveText without fileName", 30)
		return false 
	end 
	if not shared then shared = flase end 
	if not theString then theString = "" end 
	
	local path = persistence.missionDir .. fileName
	if shared then 
		-- we would now change the path
		trigger.action.outText("+++persistence: NYI: shared", 30)
		return 
	end

	local theFile = nil 
	
	if append then 
		theFile = io.open(path, "a")
	else 
		theFile = io.open(path, "w")
	end

	if not theFile then 
		trigger.action.outText("+++persistence: saveText - unable to open " .. path, 30)
		return false 
	end
	
	theFile:write(theString)
	
	theFile:close()
	
	return true 
end

function persistence.saveTable(theTable, fileName, shared, append)
	if not persistence.active then return false end 
	if not fileName then return false end
	if not theTable then return false end 
	if not shared then shared = false end 
	
	local theString = net.lua2json(theTable)
	
	if not theString then theString = "" end 
	
	local path = persistence.missionDir .. fileName
	if shared then 
		-- we would now change the path
		trigger.action.outText("+++persistence: NYI: shared", 30)
		return 
	end
	
	local theFile = nil 
	
	if append then 
		theFile = io.open(path, "a")
	else 
		theFile = io.open(path, "w")
	end

	if not theFile then 
		return false 
	end
	
	theFile:write(theString)
	
	theFile:close()
	
	return true 
end


function persistence.loadText(fileName) -- load file as text
	if not persistence.active then return nil end 
	if not fileName then return nil end
	
	local path = persistence.missionDir .. fileName
	local theFile = io.open(path, "r") 
	if not theFile then return nil end
	
	local t = theFile:read("*a")
	
	theFile:close()
	
	return t
end

function persistence.loadTable(fileName) -- load file as table 
	if not persistence.active then return nil end 
	if not fileName then return nil end

	local t = persistence.loadText(fileName)
	
	if not t then return nil end 
	
	local tab = net.json2lua(t)

	return tab
end



--
-- Data Load on Start
--
function persistence.initFlagsFromData(theFlags)
	-- assumes that theFlags is a dictionary containing 
	-- flag names 
	local flagLog = ""
	local flagCount = 0
	for flagName, value in pairs(theFlags) do 
		local val = tonumber(value) -- ensure number 
		if not val then val = 0 end 
		trigger.action.setUserFlag(flagName, val)
		if flagLog ~= "" then 
			flagLog = flagLog .. ", " .. flagName .. "=" .. val 
		else 
			flagLog = flagName .. "=" .. val 
		end
		flagCount = flagCount + 1
	end 
	if persistence.verbose and flagCount > 0 then 
		trigger.action.outText("+++persistence: loaded " .. flagCount .. " flags from storage:\n" .. flagLog .. "", 30)
	elseif persistence.verbose then
		trigger.action.outText("+++persistence: no flags loaded, commencing mission data load", 30)
	end	
	
end

function persistence.missionStartDataLoad()
	-- check one: see if we have mission data 
	local theData = persistence.loadTable(persistence.saveFileName)
	
	if not theData then 
		if persistence.verbose then 
			trigger.action.outText("+++persistence: no saved data, fresh start.", 30)
		end
		return 
	end -- there was no data to load
	
	if theData["freshMaker"] then 
		if persistence.verbose then 
			trigger.action.outText("+++persistence: detected fresh start.", 30)
		end
		return 
	end
	
	-- when we get here, we got at least some data. check it 
	if theData["versionID"] or persistence.versionID then 
		local vid = theData.versionID -- note: either may be nil!
		if vid ~= persistence.versionID then 
			-- we pretend load never happened.
			-- simply return
			if persistence.verbose then 
				local curvid = persistence.versionID
				if not curvid then curvid = "<NIL>" end 
				if not vid then vid = "<NIL>" end 
				trigger.action.outText("+++persistence: version mismatch\n(saved = <" .. vid .. "> vs current = <" .. curvid .. ">) - fresh start.", 30)
			end
			return 
		end
	end
	
	-- we have valid data, and modules, after signing up 
	-- can init from by data 
	persistence.missionData = theData
	persistence.hasData = true 
	trigger.action.setUserFlag("cfxPersistenceHasData", 1)
	
	-- init my flags from last save 
	local theFlags = theData["persistence.flagData"]
	if theFlags then 
		persistence.initFlagsFromData(theFlags)
	end
	
	-- we are done for now. modules check in 
	-- after persistence and load their own data 
	-- when they detect that there is data to load 

	trigger.action.outText("+++persistence: successfully read mission save data", 30)

end

--
-- MAIN DATA WRITE
--
function persistence.collectFlagData()
	local flagData = {}
	for idx, flagName in pairs (persistence.flagsToSave) do 
		local theNum = trigger.misc.getUserFlag(flagName)
		flagData[flagName] = theNum

	end
	return flagData
end

function persistence.saveMissionData()
	local myData = {}

	-- first, handle versionID and freshMaker
	if persistence.freshMaker then 
		myData["freshMaker"] = true 
	end
	
	if persistence.versionID then 
		myData["versionID"] = persistence.versionID 
	end
		
	-- now handle flags 
	myData["persistence.flagData"] = persistence.collectFlagData()
	
	-- now handle all other modules 
	for moduleName, callbacks in pairs(persistence.callbacks) do
		local moduleData = callbacks.persistData()
		if moduleData then 
			myData[moduleName] = moduleData
			if persistence.verbose then 
				trigger.action.outText("+++persistence: gathered data from <" .. moduleName .. ">", 30)
			end
		else 
			if persistence.verbose then 
				trigger.action.outText("+++persistence: NO DATA gathered data from <" .. moduleName .. ">, module returned NIL", 30)
			end
		end 
	end
	
	-- now save data to file 
	persistence.saveTable(myData, persistence.saveFileName)
end

--
-- UPDATE 
--
function persistence.doSaveMission()
	-- main save entry, also from API 
	if persistence.verbose then 
		trigger.action.outText("+++persistence: starting save", 30)
	end
	
	if persistence.active then 
		persistence.saveMissionData()
	else 
		if persistence.verbose then 
			trigger.action.outText("+++persistence: not actice. skipping save", 30)
		end
		return 
	end 
	
	if persistence.saveNotification then 
		trigger.action.outText("+++persistence: mission saved to\n" .. persistence.missionDir .. persistence.saveFileName, 30)
	end
end

function persistence.noteCleanRestart()
	persistence.freshMaker = true 
	persistence.doSaveMission()
	trigger.action.outText("\n\nYou can re-start the mission for a fresh start.\n\n",30)
	
end

function persistence.update()
	-- call me in a second to poll triggers
	timer.scheduleFunction(persistence.update, {}, timer.getTime() + 1/persistence.ups)
	
	-- check my trigger flag 
	if persistence.saveMission and cfxZones.testZoneFlag(persistence, persistence.saveMission, "change", "lastSaveMission") then 
		persistence.doSaveMission()
	end
	
	if persistence.cleanRestart and cfxZones.testZoneFlag(persistence, persistence.cleanRestart, "change", "lastCleanRestart") then 
		persistence.noteCleanRestart()
	end
	
	-- check my timer 
	if persistence.saveTime and persistence.saveTime < timer.getTime() then 
		persistence.doSaveMission()
		-- start next cycle 
		persistence.saveTime = persistence.saveInterval * 60 + timer.getTime()
	end
end
--
-- config & start 
--

function persistence.collectFlagsFromZone(theZone)
	local theFlags = theZone:getStringFromZoneProperty("saveFlags", "*dummy")
	persistence.registerFlagsToSave(theFlags, theZone)
end

function persistence.readConfigZone()
	if not _G["lfs"] then 
		trigger.action.outText("+++persistence: DCS correctly not 'desanitized'. Persistence disabled", 30)
		return 
	end
	
	local theZone = cfxZones.getZoneByName("persistenceConfig") 
	local hasConfig = true
	if not theZone then 
		hasConfig = false 
		theZone = cfxZones.createSimpleZone("persistenceConfig")
	end 
	
	-- serverDir is the path from the server save directory, usually "Missions/".
    -- will be added to lfs.writedir() unless given a root attribute 
	if theZone:hasProperty("root") then 
		-- we split this to enable further processing down the 
		-- line if neccessary
		persistence.root = theZone:getStringFromZoneProperty("root", lfs.writedir()) -- safe default
		if not dcsCommon.stringEndsWith(persistence.root, "\\") then 
			persistence.root = persistence.root .. "\\"
		end
		if theZone.verbose then 
			trigger.action.outText("+++persistence: setting root to <" .. persistence.root .. ">", 30)
		end
	else 
		persistence.root = lfs.writedir() -- safe defaulting
		if theZone.verbose then 
			trigger.action.outText("+++persistence: defaulting root to <" .. persistence.root .. ">", 30)
		end
	end
	
	persistence.serverDir = theZone:getStringFromZoneProperty("serverDir", "Missions\\")

	if hasConfig then 
		if theZone:hasProperty("saveDir") then 
			persistence.saveDir = theZone:getStringFromZoneProperty("saveDir", "")
		else 
			-- local missname = net.dostring_in("gui", "return DCS.getMissionName()") .. " (data)"
			persistence.saveDir = dcsCommon.getMissionName() .. " (data)"
		end
	else 
		persistence.saveDir = "" -- save dir is to main mission 
		-- so that when no config is present (standalone debugger)
		-- this will not cause a separate save folder 
	end
	
	if persistence.saveDir == "" and persistence.verbose then 
		trigger.action.outText("*** WARNING: persistence is set to write to main mission directory!", 30)
	end
	
	if theZone:hasProperty("saveFileName") then 
		persistence.saveFileName = theZone:getStringFromZoneProperty("saveFileName", dcsCommon.getMissionName() .. " Data.txt")
	end
	
	if theZone:hasProperty("versionID") then
		persistence.versionID = theZone:getStringFromZoneProperty("versionID", "") -- to check for full restart 
	end 
	
	persistence.saveInterval = theZone:getNumberFromZoneProperty("saveInterval", -1) -- default to manual save
	if persistence.saveInterval > 0 then 
		persistence.saveTime = persistence.saveInterval * 60 + timer.getTime()
	end
	
	if theZone:hasProperty("cleanRestart?") then 
		persistence.cleanRestart = theZone:getStringFromZoneProperty("cleanRestart?", "*<none>")
		persistence.lastCleanRestart = theZone:getFlagValue(persistence.cleanRestart)
	end
	
	if theZone:hasProperty("saveMission?") then 
		persistence.saveMission = theZone:getStringFromZoneProperty("saveMission?", "*<none>")
		persistence.lastSaveMission = theZone:getFlagValue(persistence.saveMission)
	end
	
	persistence.verbose = theZone.verbose
	
	persistence.saveNotification = theZone:getBoolFromZoneProperty("saveNotification", true)
	
	if persistence.verbose then 
		trigger.action.outText("+++persistence: read config", 30)
	end 
	
end

function persistence.start()
	-- lib check 
	if not dcsCommon.libCheck then 
		trigger.action.outText("persistence requires dcsCommon", 30)
		return false 
	end 
	if not dcsCommon.libCheck("persistence", persistence.requiredLibs) then
		return false 
	end
		
	-- read config 
	persistence.saveFileName = dcsCommon.getMissionName() .. " Data.txt"
	persistence.readConfigZone()
	
	-- let's see it lfs and io are online 
	persistence.active = false 
	if (not _G["lfs"]) or (not lfs) then 
		if persistence.verbose then 
			trigger.action.outText("+++persistence requires 'lfs'", 30)
		end
		return false
	end
	if not _G["io"] then 
		if persistence.verbose then 
			trigger.action.outText("+++persistence requires 'io'", 30)
		end
		return false
	end
	
	local mainDir = persistence.root .. persistence.serverDir 
	if not dcsCommon.stringEndsWith(mainDir, "\\") then 
		mainDir = mainDir .. "\\"
	end

	-- lets see if we can access the server's mission directory and 
	-- save directory 	
	if persistence.isDir(mainDir) then 
		if persistence.verbose then 
			trigger.action.outText("persistence: main dir is <" .. mainDir .. ">", 30)
		end
	else 
		if persistence.verbose then 
			trigger.action.outText("+++persistence: Main directory <" .. mainDir .. "> not found or not a directory", 30)
		end 
		return false 
	end	
	persistence.mainDir = mainDir

	local missionDir = mainDir .. persistence.saveDir
	if not dcsCommon.stringEndsWith(missionDir, "\\") then 
		missionDir = missionDir .. "\\"
	end
	
	-- check if mission dir exists already 
	local success, mode = persistence.hasFile(missionDir)
	if success and mode == "directory" then 
		-- has been allocated, and is dir
		if persistence.verbose then 
			trigger.action.outText("+++persistence: saving mission data to <" .. missionDir .. ">", 30)
		end
	elseif success then 
		if persistence.verbose then 
			trigger.action.outText("+++persistence: <" .. missionDir .. "> is not a directory", 30)
		end
		return false 
	else 
		-- does not exist, try to allocate it
		if persistence.verbose then 
			trigger.action.outText("+++persistence: will now create <" .. missionDir .. ">", 30)
		end		
		local ok, mkErr = lfs.mkdir(missionDir)
		if not ok then 
			if persistence.verbose then 
				trigger.action.outText("+++persistence: unable to create <" .. missionDir .. ">: <" .. mkErr .. ">", 30)
			end
			return false
		end
		if persistence.verbose then 
			trigger.action.outText("+++persistence: created <" .. missionDir .. "> successfully, will save mission data here", 30)
		end 
	end
	
	-- missionDir is root + serverDir + saveDir 
	persistence.missionDir = missionDir
	
	persistence.active = true -- we can load and save data 
	trigger.action.setUserFlag("cfxPersistence", 1)
    persistence.hasData = false -- we do not have save data 
	
	-- from here on we can read and write files in the missionDir 	
	-- read persistence attributes from all zones 
	local attrZones = cfxZones.getZonesWithAttributeNamed("saveFlags")
	for k, aZone in pairs(attrZones) do 
		persistence.collectFlagsFromZone(aZone) -- process attributes
		-- we do not retain the zone, it's job is done
	end

	if persistence.verbose then 
		trigger.action.outText("+++persistence is active", 30)
	end	
	
	-- we now see if we can and need load data 
	persistence.missionStartDataLoad()
	
	-- and start updating 
	persistence.update()
	
	return persistence.active 
end

--
-- go!
--

if not persistence.start() then 
	if persistence.verbose then 
		trigger.action.outText("+++ persistence not available", 30)
	end
	-- we do NOT remove the methods so we don't crash 
end


-- theDebugger 
-- theDebugger 2.x
debugger = {}
debugger.version = "2.1.0"
debugDemon = {}
debugDemon.version = "2.1.0"

debugger.verbose = false 
debugger.ups = 4 -- every 0.25 second  
debugger.name = "DML Debugger" -- for compliance with cfxZones 

debugger.log = ""

--[[--
	Version History
	2.0.0 - dmlZones OOP 
	      - eventmon command 
		  - eventmon all, off, event #
		  - standard events 
		  - adding events via #
		  - events? attribute from any zone 
		  - eventmon last command 
		  - q - query MSE Lua variables 
		  - w - write/overwrite MSE Lua variables 
		  - a - analyse Lua tables / variables 
		  - smoke
		  - spawn system with predefines
		  - spawn coalition
		  - spawn number 
		  - spawn heading
		  - spawn types 
		  - spawn aircraft: add waypoints 
		  - spawn "?"
		  - debuggerSpawnTypes zone 
		  - reading debuggerSpawnTypes 
		  - removed some silly bugs / inconsistencies
	2.1.0 - debugging code is now invoked deferred to avoid 
	        DCS crash after exiting. Debug code now executes 
			outside of the event code's bracket.
			debug invocation on clone of data structure 
			readback verification of flag set 
			fixed getProperty() in debugger with zone 
			
--]]--

debugger.requiredLibs = {
	"dcsCommon", -- always
	"cfxZones", -- Zones, of course 
}
-- note: saving logs requires persistence module 
-- will auto-abort saving if not present 


debugger.debugZones = {}
debugger.debugUnits = {}
debugger.debugGroups = {}
debugger.debugObjects = {}
debugger.showEvents = {}
debugger.lastEvent = nil 

debugDemon.eventList = {
  ["0"] = "S_EVENT_INVALID = 0",
  ["1"] = "S_EVENT_SHOT = 1",
  ["2"] = "S_EVENT_HIT = 2",
  ["3"] = "S_EVENT_TAKEOFF = 3",
  ["4"] = "S_EVENT_LAND = 4",
  ["5"] = "S_EVENT_CRASH = 5",
  ["6"] = "S_EVENT_EJECTION = 6",
  ["7"] = "S_EVENT_REFUELING = 7",
  ["8"] = "S_EVENT_DEAD = 8",
  ["9"] = "S_EVENT_PILOT_DEAD = 9",
  ["10"] = "S_EVENT_BASE_CAPTURED = 10",
  ["11"] = "S_EVENT_MISSION_START = 11",
  ["12"] = "S_EVENT_MISSION_END = 12",
  ["13"] = "S_EVENT_TOOK_CONTROL = 13",
  ["14"] = "S_EVENT_REFUELING_STOP = 14",
  ["15"] = "S_EVENT_BIRTH = 15",
  ["16"] = "S_EVENT_HUMAN_FAILURE = 16",
  ["17"] = "S_EVENT_DETAILED_FAILURE = 17",
  ["18"] = "S_EVENT_ENGINE_STARTUP = 18",
  ["19"] = "S_EVENT_ENGINE_SHUTDOWN = 19",
  ["20"] = "S_EVENT_PLAYER_ENTER_UNIT = 20",
  ["21"] = "S_EVENT_PLAYER_LEAVE_UNIT = 21",
  ["22"] = "S_EVENT_PLAYER_COMMENT = 22",
  ["23"] = "S_EVENT_SHOOTING_START = 23",
  ["24"] = "S_EVENT_SHOOTING_END = 24",
  ["25"] = "S_EVENT_MARK_ADDED  = 25", 
  ["26"] = "S_EVENT_MARK_CHANGE = 26",
  ["27"] = "S_EVENT_MARK_REMOVED = 27",
  ["28"] = "S_EVENT_KILL = 28",
  ["29"] = "S_EVENT_SCORE = 29",
  ["30"] = "S_EVENT_UNIT_LOST = 30",
  ["31"] = "S_EVENT_LANDING_AFTER_EJECTION = 31",
  ["32"] = "S_EVENT_PARATROOPER_LENDING = 32",
  ["33"] = "S_EVENT_DISCARD_CHAIR_AFTER_EJECTION = 33", 
  ["34"] = "S_EVENT_WEAPON_ADD = 34",
  ["35"] = "S_EVENT_TRIGGER_ZONE = 35",
  ["36"] = "S_EVENT_LANDING_QUALITY_MARK = 36",
  ["37"] = "S_EVENT_BDA = 37", 
  ["38"] = "S_EVENT_AI_ABORT_MISSION = 38", 
  ["39"] = "S_EVENT_DAYNIGHT = 39", 
  ["40"] = "S_EVENT_FLIGHT_TIME = 40", 
  ["41"] = "S_EVENT_PLAYER_SELF_KILL_PILOT = 41", 
  ["42"] = "S_EVENT_PLAYER_CAPTURE_AIRFIELD = 42", 
  ["43"] = "S_EVENT_EMERGENCY_LANDING = 43",
  ["44"] = "S_EVENT_UNIT_CREATE_TASK = 44",
  ["45"] = "S_EVENT_UNIT_DELETE_TASK = 45",
  ["46"] = "S_EVENT_SIMULATION_START = 46",
  ["47"] = "S_EVENT_WEAPON_REARM = 47",
  ["48"] = "S_EVENT_WEAPON_DROP = 48",
  ["49"] = "S_EVENT_UNIT_TASK_TIMEOUT = 49",
  ["50"] = "S_EVENT_UNIT_TASK_STAGE = 50",
  ["51"] = "S_EVENT_MAC_SUBTASK_SCORE = 51", 
  ["52"] = "S_EVENT_MAC_EXTRA_SCORE = 52",
  ["53"] = "S_EVENT_MISSION_RESTART = 53",
  ["54"] = "S_EVENT_MISSION_WINNER = 54", 
  ["55"] = "S_EVENT_POSTPONED_TAKEOFF = 55", 
  ["56"] = "S_EVENT_POSTPONED_LAND = 56", 
  ["57"] = "S_EVENT_MAX = 57",
}

debugger.spawnTypes = {
 ["inf"] = "Soldier M4",
 ["ifv"] = "BTR-80",
 ["tank"] = "T-90",
 ["ship"] = "PERRY",
 ["helo"] = "AH-1W",
 ["jet"] = "MiG-21Bis",
 ["awacs"] = "A-50",
 ["ww2"] = "SpitfireLFMkIX",
 ["bomber"] = "B-52H",
 ["cargo"] = "ammo_cargo",
 ["sam"] = "Roland ADS",
 ["aaa"] = "ZSU-23-4 Shilka",
 ["arty"] = "M-109",
 ["truck"] = "KAMAZ Truck",
 ["drone"] = "MQ-9 Reaper",
 ["manpad"] = "Soldier stinger",
 ["obj"] = "house2arm"
}
--
-- Logging & saving 
--

function debugger.outText(message, seconds, cls)
	if not message then message = "" end 
	if not seconds then seconds = 20 end 
	if not cls then cls = false end 
	
	-- append message to log, and add a lf
	if not debugger.log then debugger.log = "" end 
	debugger.log = debugger.log .. message .. "\n"
	
	-- now hand up to trigger 
	trigger.action.outText(message, seconds, cls)
end

function debugger.saveLog(name)
	if not _G["persistence"] then 
		debugger.outText("+++debug: persistence module required to save log")
		return
	end
	
	if not persistence.active then 
		debugger.outText("+++debug: persistence module can't write. ensur you desanitize lfs and io")
		return 
	end
	
	if persistence.saveText(debugger.log, name) then 
		debugger.outText("+++debug: log saved to <" .. persistence.missionDir .. name .. ">")
	else 
		debugger.outText("+++debug: unable to save log to <" .. persistence.missionDir .. name .. ">")
	end
end


--
-- tracking flags 
--

function debugger.addDebugger(theZone)
	table.insert(debugger.debugZones, theZone)
end

function debugger.getDebuggerByName(aName) 
	for idx, aZone in pairs(debugger.debugZones) do 
		if aName == aZone.name then return aZone end 
	end
	if debugger.verbose then 
		debugger.outText("+++debug: no debug zone with name <" .. aName ..">", 30)
	end 
	
	return nil 
end

function debugger.removeDebugger(theZone)
	local filtered = {}
	for idx, dZone in pairs(debugger.debugZones) do 
		if dZone == theZone then 
		else 
			table.insert(filtered, dZone)
		end
	end
	debugger.debugZones = filtered
end
--
-- read zone 
-- 
function debugger.createDebuggerWithZone(theZone)
	-- watchflag input trigger
	theZone.debugInputMethod = theZone:getStringFromZoneProperty( "triggerMethod", "change")
	if theZone:hasProperty("debugTriggerMethod") then 
		theZone.debugInputMethod = theZone:getStringFromZoneProperty("debugTriggerMethod", "change")
	elseif theZone:hasProperty("inputMethod") then 
		theZone.debugInputMethod = theZone:getStringFromZoneProperty(theZone, "inputMethod", "change")
	elseif theZone:hasProperty("sayWhen") then 
		theZone.debugInputMethod = theZone:getStringFromZoneProperty("sayWhen", "change")
	end
	
	-- say who we are and what we are monitoring
	if debugger.verbose or theZone.verbose then 
		debugger.outText("---debug: adding zone <".. theZone.name .."> to look for <value " .. theZone.debugInputMethod .. "> in flag(s):", 30)
	end
	
	-- read main debug array
	local theFlags = theZone:getStringFromZoneProperty("debug?", "<none>")
	-- now, create an array from that
	local flagArray = cfxZones.flagArrayFromString(theFlags)
	local valueArray = {}
	-- now establish current values 
	for idx, aFlag in pairs(flagArray) do 
		local fVal = theZone:getFlagValue(aFlag)
		if debugger.verbose or theZone.verbose then 
			debugger.outText("    monitoring flag <" .. aFlag .. ">, inital value is <" .. fVal .. ">", 30)
		end
		valueArray[aFlag] = fVal
	end
	theZone.flagArray = flagArray
	theZone.valueArray = valueArray 
	
	-- DML output method
	theZone.debugOutputMethod = theZone:getStringFromZoneProperty("method", "inc")
	if theZone:hasProperty("outputMethod") then 
		theZone.debugOutputMethod = theZone:getStringFromZoneProperty("outputMethod", "inc")
	end
	if theZone:hasProperty("debugMethod") then 
		theZone.debugOutputMethod = theZone:getStringFromZoneProperty("debugMethod", "inc")
	end
	
	-- notify!
	if theZone:hasProperty("notify!") then 
		theZone.debugNotify = theZone:getStringFromZoneProperty("notify!", "<none>")
	end
	
	-- debug message, can use all messenger vals plus <f> for flag name 
	-- we use out own default
	-- with <f> meaning flag name, <p> previous value, <c> current value 
	theZone.debugMsg = theZone:getStringFromZoneProperty("debugMsg", "---debug: <t> -- Flag <f> changed from <p> to <c> [<z>]")
end

function debugger.createEventMonWithZone(theZone)
	local theFlags = theZone:getStringFromZoneProperty("events?", "<none>")
	local flagArray = cfxZones.flagArrayFromString(theFlags)
	local valueArray = {}
	-- now establish current values 
	if debugger.verbose or theZone.verbose then 
		debugger.outText("*** monitoring events defined in <" .. theZone.name .. ">:", 30)
	end
	for idx, aFlag in pairs(flagArray) do 
		local evt = tonumber(aFlag) 		
		if evt and (debugger.verbose or theZone.verbose) then 
			if evt < 0 then evt = 0 end 
			if evt > 57 then evt = 57 end 
			debugger.showEvents[evt] = debugDemon.eventList[tostring(evt)]
			debugger.outText("    monitoring event <" .. debugger.showEvents[evt] .. ">", 30)
		end
	end
end

--
-- Misc
--
function debugger.addFlagToObserver(flagName, theZone)
	table.insert(theZone.flagArray, flagName)
	local fVal = cfxZones.getFlagValue(flagName, theZone)
	theZone.valueArray[flagName] = fVal
end

function debugger.removeFlagFromObserver(flagName, theZone)
	local filtered = {}
	for idy, aName in pairs(theZone.flagArray) do 
		if aName == flagName then
		else
			table.insert(filtered, aName)
		end 
	end
	theZone.flagArray = filtered 
	-- no need to clean up values, they are name-indexed. do it anyway
	theZone.valueArray[flagName] = nil
end

function debugger.isObservingWithObserver(flagName, theZone)
	for idy, aName in pairs(theZone.flagArray) do 
		if aName == flagName then
			local val = theZone.valueArray[flagName]
			return true, val 
		end 	
	end	
end

function debugger.isObserving(flagName)
	-- scan all zones for flag, and return 
	-- zone, and flag value if observing
	local observers = {}
	for idx, theZone in pairs(debugger.debugZones) do 
		for idy, aName in pairs(theZone.flagArray) do 
			if aName == flagName then
				table.insert(observers, theZone)
			end 
		end
	end
	return observers 
end

--
-- Update 
--
function debugger.processDebugMsg(inMsg, theZone, theFlag, oldVal, currVal)
	if not inMsg then return "<nil inMsg>" end
	if not oldVal then oldVal = "<no val!>" else oldVal = tostring(oldVal) end 
	if not currVal then currVal = "<no val!>" else currVal = tostring(currVal) end 
	if not theFlag then theFlag = "<no flag!>" end 
	
	local formerType = type(inMsg)
	if formerType ~= "string" then inMsg = tostring(inMsg) end  
	if not inMsg then inMsg = "<inMsg is incompatible type " .. formerType .. ">" end 
	
	-- build message by relacing wildcards
	local outMsg = ""
	
	-- replace line feeds 
	outMsg = inMsg:gsub("<n>", "\n")
	if theZone then 
		outMsg = outMsg:gsub("<z>", theZone.name)
	end
	
	-- replace <C>, <p>, <f> with currVal, oldVal, flag
	outMsg = outMsg:gsub("<c>", currVal)
	outMsg = outMsg:gsub("<p>", oldVal)
	outMsg = outMsg:gsub("<o>", oldVal) -- just for QoL
	outMsg = outMsg:gsub("<f>", theFlag)
	
	-- replace <t> with current mission time HMS
	local absSecs = timer.getAbsTime()-- + env.mission.start_time
	while absSecs > 86400 do 
		absSecs = absSecs - 86400 -- subtract out all days 
	end
	local timeString  = dcsCommon.processHMS("<:h>:<:m>:<:s>", absSecs)
	outMsg = outMsg:gsub("<t>", timeString)
	
	-- replace <lat> with lat of zone point and <lon> with lon of zone point 
	-- and <mgrs> with mgrs coords of zone point 
	if theZone then 
		local currPoint = cfxZones.getPoint(theZone)
		local lat, lon, alt = coord.LOtoLL(currPoint)
		lat, lon = dcsCommon.latLon2Text(lat, lon)
		outMsg = outMsg:gsub("<lat>", lat)
		outMsg = outMsg:gsub("<lon>", lon)
		currPoint = cfxZones.getPoint(theZone)
		local grid = coord.LLtoMGRS(coord.LOtoLL(currPoint))
		local mgrs = grid.UTMZone .. ' ' .. grid.MGRSDigraph .. ' ' .. grid.Easting .. ' ' .. grid.Northing
		outMsg = outMsg:gsub("<mgrs>", mgrs)
	end 
	
	return outMsg
end

function debugger.debugZone(theZone)
	-- check every flag of this zone 
	for idx, aFlag in pairs(theZone.flagArray) do 
		local oldVal = theZone.valueArray[aFlag]
		local oldVal = theZone.valueArray[aFlag]
		theZone.debugLastVal = oldVal
		local hasChanged, newValue = cfxZones.testZoneFlag(
			theZone,
			aFlag, 
			theZone.debugInputMethod,
 			"debugLastVal")
		-- we ALWAYS transfer latch back
		theZone.valueArray[aFlag] = newValue
		
		if hasChanged then 
			-- we are triggered 
			-- generate the ouput message
			local msg = theZone.debugMsg
			msg = debugger.processDebugMsg(msg, theZone, aFlag, oldVal, newValue)
			debugger.outText(msg, 30)
		end
	end

end

--
-- reset debugger
--
function debugger.resetObserver(theZone)
	for idf, aFlag in pairs(theZone.flagArray) do 
		local fVal = cfxZones.getFlagValue(aFlag, theZone)
		if debugger.verbose or theZone.verbose then 
			debugger.outText("---debug: resetting flag <" .. aFlag .. ">, to <" .. fVal .. "> for zone <" .. theZone.name .. ">", 30)
		end
		theZone.valueArray[aFlag] = fVal
	end	
end

function debugger.reset()
	for idx, theZone in pairs(debugger.debugZones) do
		-- reset this zone 
		debugger.resetObserver(theZone)
	end
end

function debugger.showObserverState(theZone)
	for idf, aFlag in pairs(theZone.flagArray) do 
		local fVal = cfxZones.getFlagValue(aFlag, theZone)
		if debugger.verbose or theZone.verbose then 
			debugger.outText("     state of flag <" .. aFlag .. ">: <" .. theZone.valueArray[aFlag] .. ">", 30)
		end
		theZone.valueArray[aFlag] = fVal
	end
end

function debugger.showState()
	debugger.outText("---debug: CURRENT STATE <" .. dcsCommon.nowString() .. "> --- ", 30)
	for idx, theZone in pairs(debugger.debugZones) do
		-- show this zone's state
		if #theZone.flagArray > 0 then 
			debugger.outText("   state of observer <" .. theZone.name .. "> looking for <value " .. theZone.debugInputMethod .. ">:", 30)
			debugger.showObserverState(theZone)
		else 
			if theZone.verbose or debugger.verbose then 
				debugger.outText("   (empty observer <" .. theZone.name .. ">)", 30)
			end
		end
	end
	debugger.outText("---debug: end of state --- ", 30)
end

function debugger.doActivate()
	debugger.active = true
	if debugger.verbose or true then 
		debugger.outText("+++ DM Debugger is now active", 30)
	end 
end

function debugger.doDeactivate()
	debugger.active = false
	if debugger.verbose or true then 
		debugger.outText("+++ debugger deactivated", 30)
	end 
end

function debugger.update()
	-- call me in a second to poll triggers
	timer.scheduleFunction(debugger.update, {}, timer.getTime() + 1/debugger.ups)
	
	-- first check for switch on or off
	if debugger.onFlag then 
		if cfxZones.testZoneFlag(debugger, debugger.onFlag, "change","lastOn") then
			debugger.doActivate()
		end
	end
	
	if debugger.offFlag then 
		if cfxZones.testZoneFlag(debugger, debugger.offFlag, "change","lastOff") then
			debugger.doDeactivate()
		end
	end
	
	-- ALWAYS check for reset & state. 
	if debugger.resetFlag then 
		if cfxZones.testZoneFlag(debugger, debugger.resetFlag, "change","lastReset") then
			debugger.reset()
		end
	end
	
	if debugger.stateFlag then 
		if cfxZones.testZoneFlag(debugger, debugger.stateFlag, "change","lastState") then
			debugger.showState()
		end
	end
	
	-- only progress if we are on
	if not debugger.active then return end 
	
	for idx, aZone in pairs(debugger.debugZones) do
		-- see if we are triggered 
		debugger.debugZone(aZone)
	end
end


--
-- Config & Start
--
function debugger.readConfigZone()
	local theZone = cfxZones.getZoneByName("debuggerConfig") 
	if not theZone then 
		theZone = cfxZones.createSimpleZone("debuggerConfig") 
	end 
	debugger.configZone = theZone 
	
	debugger.active = theZone:getBoolFromZoneProperty("active", true)
	debugger.verbose = theZone.verbose 
	
	if theZone:hasProperty("on?") then 
		debugger.onFlag = theZone:getStringFromZoneProperty("on?", "<none>")
		debugger.lastOn = cfxZones.getFlagValue(debugger.onFlag, theZone)
	end 

	if theZone:hasProperty("off?") then 
		debugger.offFlag = theZone:getStringFromZoneProperty("off?", "<none>")
		debugger.lastOff = cfxZones.getFlagValue(debugger.offFlag, theZone)
	end 

	if theZone:hasProperty("reset?") then 
		debugger.resetFlag = theZone:getStringFromZoneProperty("reset?", "<none>")
		debugger.lastReset = cfxZones.getFlagValue(debugger.resetFlag, theZone)
	end
	
	if theZone:hasProperty("state?") then 
		debugger.stateFlag = theZone:getStringFromZoneProperty("state?", "<none>")
		debugger.lastState = cfxZones.getFlagValue(debugger.stateFlag, theZone)
	end
	
	debugger.ups = theZone:getNumberFromZoneProperty("ups", 4)
end

function debugger.readSpawnTypeZone()
	local theZone = cfxZones.getZoneByName("debuggerSpawnTypes") 
	if not theZone then 
		theZone = cfxZones.createSimpleZone("debuggerSpawnTypes") 
	end 
	local allAttribuites = theZone:getAllZoneProperties()
	for attrName, aValue in pairs(allAttribuites) do 
		local theLow = string.lower(attrName)
		local before = debugger.spawnTypes[theLow]
		if before then 
			debugger.spawnTypes[theLow] = aValue
			if theZone.verbose or debugger.verbose then 
				trigger.action.outText("+++debug: changed generic '" .. theLow .. "' from <" .. before .. "> to <" .. aValue .. ">", 30)
			end
		else 
			if theZone.verbose or debugger.verbose then
				if theLow == "verbose" then -- filtered 
				else 
					trigger.action.outText("+++debug: generic '" .. theLow .. "' unknown, not replaced.", 30)
				end
			end 
		end 
	end 
end 


function debugger.start()
	-- lib check
	if not dcsCommon.libCheck then 
		trigger.action.outText("cfx debugger requires dcsCommon", 30)
		return false 
	end 
	if not dcsCommon.libCheck("cfx debugger", debugger.requiredLibs) then
		return false 
	end
	
	-- read config 
	debugger.readConfigZone()
	
	-- read spawn types 
	debugger.readSpawnTypeZone() 
	
	-- process debugger Zones 
	-- old style
	local attrZones = cfxZones.getZonesWithAttributeNamed("debug?")
	for k, aZone in pairs(attrZones) do 
		debugger.createDebuggerWithZone(aZone) -- process attributes
		debugger.addDebugger(aZone) -- add to list
	end
	
	local attrZones = cfxZones.getZonesWithAttributeNamed("debug")
	for k, aZone in pairs(attrZones) do 
		debugger.outText("***Warning: Zone <" .. aZone.name .. "> has a 'debug' attribute. Are you perhaps missing a '?'", 30)
	end
	
	local attrZones = cfxZones.getZonesWithAttributeNamed("events?")
	for k, aZone in pairs(attrZones) do 
		debugger.createEventMonWithZone(aZone) -- process attributes
	end
	
	local attrZones = cfxZones.getZonesWithAttributeNamed("events")
	for k, aZone in pairs(attrZones) do 
		debugger.outText("***Warning: Zone <" .. aZone.name .. "> has an 'events' attribute. Are you perhaps missing a '?'", 30)
	end
	-- events 
	
	-- say if we are active
	if debugger.verbose then 
		if debugger.active then 
			debugger.outText("+++debugger loaded and active", 30)
		else 
			debugger.outText("+++ debugger: standing by for activation", 30)
		end
	end
	
	-- start update 
	debugger.update()
	
	debugger.outText("cfx debugger v" .. debugger.version .. " started.", 30)
	return true 
end

-- let's go!
if not debugger.start() then 
	trigger.action.outText("cfx debugger aborted: missing libraries", 30)
	debugger = nil 
end


--
-- DEBUG DEMON 
--

debugDemon.myObserverName = "+DML Debugger+"
-- interactive interface for DML debugger 

debugDemon.verbose = false 
-- based on cfx stage demon
--[[--
	Version History
	1.0.0 - initial version 
	1.1.0 - save command, requires persistence
	2.0.0 - eventmon 
	      - dml zones OOP 
--]]--

debugDemon.requiredLibs = {
	"dcsCommon", -- always
	"cfxZones", -- Zones, of course 
	"debugger",
}
debugDemon.markOfDemon = "-" -- all commands must start with this sequence
debugDemon.splitDelimiter = " " 
debugDemon.commandTable = {} -- key, value pair for command processing per keyword
debugDemon.keepOpen = false -- keep mark open after a successful command
debugDemon.snapshot = {}
debugDemon.activeIdx = -1 -- to detect if a window was close 
						  -- and prevent execution of debugger 
function debugDemon.hasMark(theString) 
	-- check if the string begins with the sequece to identify commands 
	if not theString then return false end
	return theString:find(debugDemon.markOfDemon) == 1
end


-- main hook into DCS. Called whenever a Mark-related event happens
-- very simple: look if text begins with special sequence, and if so, 
-- call the command processor. 
function debugDemon:onEvent(theEvent)
	-- first order of business: call the event monitor 
	debugDemon.doEventMon(theEvent)
	
	-- now process our own 
	-- while we can hook into any of the three events, 
	-- we curently only utilize CHANGE Mark
	if not (theEvent.id == world.event.S_EVENT_MARK_ADDED) and
	   not (theEvent.id == world.event.S_EVENT_MARK_CHANGE) and 
	   not (theEvent.id == world.event.S_EVENT_MARK_REMOVED) then 
		-- not of interest for us, bye bye
		return 
	end
						   
	-- when we get here, we have a mark event
	
    if theEvent.id == world.event.S_EVENT_MARK_ADDED then
		-- add mark is quite useless		
	end
    
    if theEvent.id == world.event.S_EVENT_MARK_CHANGE then
--		trigger.action.outText("debugger: Mark Change event received", 30)
		-- when changed, the mark's text is examined for a command
		-- if it starts with the 'mark' string ("-" by  default) it is processed
		-- by the command processor
		-- if it is processed succesfully, the mark is immediately removed
		-- else an error is displayed and the mark remains.
		if debugDemon.hasMark(theEvent.text) then 
			-- strip the mark 
			local cCommand = dcsCommon.clone(theEvent.text, true)
			local commandString = cCommand:sub(1+debugDemon.markOfDemon:len())
			-- break remainder apart into <command> <arg1> ... <argn>
			local commands = dcsCommon.splitString(commandString, debugDemon.splitDelimiter)

			-- this is a command. process it and then remove it if it was executed successfully
			local cTheEvent = dcsCommon.clone(theEvent, true) -- strip meta tables
			local args = {commands, cTheEvent}	
			-- defer execution for 0.1s to get out of trx bracked
			timer.scheduleFunction(debugDemon.deferredDebug, args, timer.getTime() + 0.1)
			debugDemon.activeIdx = cTheEvent.idx 
			--[[--
			local success = debugDemon.executeCommand(commands, cTheEvent) -- execute on a clone, not original 
						
			-- remove this mark after successful execution
			if success then 
				trigger.action.removeMark(theEvent.idx) 
			else 
				-- we could play some error sound
			end
			--]]--
		end 
    end 
	
	if theEvent.id == world.event.S_EVENT_MARK_REMOVED then
--		trigger.action.outText("Mark Remove received, removing idx <" .. theEvent.idx .. ">.", 30)
		debugDemon.activeIdx = nil 
    end
end

function debugDemon.deferredDebug(args)
--	trigger.action.outText("enter deferred debug command", 30)
--	if not debugDemon.activeIdx then 
--		trigger.action.outText("Debugger: window was closed, debug command ignored.", 30)
--		return 
--	end 
	local commands = args[1]
	local cTheEvent = args[2]
	local success = debugDemon.executeCommand(commands, cTheEvent) -- execute on a clone, not original 
				
	-- remove this mark after successful execution
	if success then 
		trigger.action.removeMark(cTheEvent.idx) 
		debugDemon.activeIdx = nil 
	else 
		-- we could play some error sound
	end
end 

--
-- add / remove commands to/from vocabulary
-- 
function debugDemon.addCommndProcessor(command, processor)
	debugDemon.commandTable[command:upper()] = processor 
end

function debugDemon.removeCommandProcessor(command)
	debugDemon.commandTable[command:upper()] = nil 
end

--
-- process input arguments. Here we simply move them 
-- up by one.
--
function debugDemon.getArgs(theCommands) 
	local args = {}
	for i=2, #theCommands do 
		table.insert(args, theCommands[i])
	end
	return args
end

--
-- debug demon's main command interpreter. 
-- magic lies in using the keywords as keys into a 
-- function table that holds all processing functions
-- I wish we had that back in the Oberon days. 
--
function debugDemon.executeCommand(theCommands, event)
	local cmd = theCommands[1]
	local arguments = debugDemon.getArgs(theCommands)
	if not cmd then return false end
	
	-- since we have a command in cmd, we remove this from
	-- the string, and pass the remainder back 
	local remainder = event.text:sub(1 + debugDemon.markOfDemon:len())
	remainder = dcsCommon.stripLF(remainder)
	remainder = dcsCommon.trim(remainder)
	remainder = remainder:sub(1+cmd:len())
	remainder = dcsCommon.trim(remainder) 

	event.remainder = remainder
	-- use the command as index into the table of functions
	-- that handle them.
	if debugDemon.commandTable[cmd:upper()] then 
		local theInvoker = debugDemon.commandTable[cmd:upper()]
		local success = theInvoker(arguments, event)
		return success
	else 
		debugger.outText("***error: unknown command <".. cmd .. ">", 30)
		return false
	end
	
	return true
end

--
-- Helpers 
--

function debugDemon.createObserver(aName)
	local observer = cfxZones.createSimpleZone(aName)
	observer.verbose = debugDemon.verbose 
	observer.debugInputMethod = "change"
	observer.flagArray = {}
	observer.valueArray = {}
	observer.debugMsg = "---debug: <t> -- Flag <f> changed from <p> to <c> [<z>]"
	return observer
end
--
-- COMMANDS
--
function debugDemon.processHelpCommand(args, event)
debugger.outText("*** debugger: commands are:" ..
	"\n  " .. debugDemon.markOfDemon .. "show <flagname/observername> -- show current values for flag or observer" ..
	"\n  " .. debugDemon.markOfDemon .. "set <flagname> <number> -- set flag to value <number>" ..
	"\n  " .. debugDemon.markOfDemon .. "inc <flagname> -- increase flag by 1, changing it" ..
	"\n  " .. debugDemon.markOfDemon .. "flip <flagname> -- when flag's value is 0, set it to 1, else to 0" ..	

	"\n\n  " .. debugDemon.markOfDemon .. "observe <flagname> [with <observername>] -- observe a flag for change" ..
	"\n  " .. debugDemon.markOfDemon .. "o <flagname> [with <observername>] -- observe a flag for change" ..
	"\n  " .. debugDemon.markOfDemon .. "forget <flagname> [with <observername>] -- stop observing a flag" ..
	"\n  " .. debugDemon.markOfDemon .. "new <observername> [[for] <condition>] -- create observer for flags" ..
	"\n  " .. debugDemon.markOfDemon .. "update <observername> [to] <condition> -- change observer's condition" ..
	"\n  " .. debugDemon.markOfDemon .. "drop <observername> -- remove observer from debugger" ..
	"\n  " .. debugDemon.markOfDemon .. "list [<match>] -- list observers [name contains <match>]" ..
	"\n  " .. debugDemon.markOfDemon .. "who <flagname> -- all who observe <flagname>" ..
	"\n  " .. debugDemon.markOfDemon .. "reset [<observername>] -- reset all or only the named observer" ..

	"\n\n  " .. debugDemon.markOfDemon .. "snap [<observername>] -- create new snapshot of flags" ..
	"\n  " .. debugDemon.markOfDemon .. "compare -- compare snapshot flag values with current" ..
	"\n  " .. debugDemon.markOfDemon .. "note <your note> -- add <your note> to the text log" ..
	"\n\n  " .. debugDemon.markOfDemon .. "spawn [<number>] [<coalition>] <type> [heading=<number>] | [?] -- spawn" .. 
	"\n               units/aircraft/objects (? for help)" ..
	"\n  " .. debugDemon.markOfDemon .. "remove <group/unit/object name> -- remove named item from mission" ..
	"\n  " .. debugDemon.markOfDemon .. "smoke <color> -- place colored smoke on the ground" ..
	"\n  " .. debugDemon.markOfDemon .. "boom <number> -- place explosion of strenght <number> on the ground" ..
	
	"\n\n  " .. debugDemon.markOfDemon .. "eventmon [all | off | <number> | ?] -- show events for all | none | event <number> | list" ..
	"\n  " .. debugDemon.markOfDemon .. "eventmon last -- analyse last reported event" ..
	"\n\n  " .. debugDemon.markOfDemon .. "q <Lua Var> -- Query value of Lua variable <Lua Var>" ..
	"\n  " .. debugDemon.markOfDemon .. "a <Lua Var> -- Analyse structure of Lua variable <Lua Var>" ..
	"\n  " .. debugDemon.markOfDemon .. "w <Lua Var> [=] <Lua Value> -- Write <Lua Value> to variable <Lua Var>" ..
	"\n\n  " .. debugDemon.markOfDemon .. "start -- starts debugger" ..
	"\n  " .. debugDemon.markOfDemon .. "stop -- stop debugger" ..

	"\n\n  " .. debugDemon.markOfDemon .. "save [<filename>] -- saves debugger log to storage" ..

	"\n\n  " .. debugDemon.markOfDemon .. "? or -help  -- this text", 30)
	return true 
end

function debugDemon.processNewCommand(args, event)
	-- syntax new <observername> [[for] <condition>]
	local observerName = args[1]
	if not observerName then 
		debugger.outText("*** new: missing observer name.", 30)
		return false -- allows correction 
	end
	
	-- see if this observer already existst
	local theObserver = debugger.getDebuggerByName(observerName)
	if theObserver then 
		debugger.outText("*** new: observer <" .. observerName .. "> already exists.", 30)
		return false -- allows correction 
	end
	
	-- little pitfall: what if the name contains blanks?
	-- also check against remainder!
	local remainderName = event.remainder
	local rObserver = debugger.getDebuggerByName(remainderName)
	if rObserver then 
		debugger.outText("*** new: observer <" .. remainderName .. "> already exists.", 30)
		return false -- allows correction 
	end
	
	local theZone = nil
	local condition = args[2] -- may need remainder instead
	if condition == "for" then 
		-- we use arg[2]
	else 
		observerName = remainderName -- we use entire rest of line 
	end

	theZone = debugDemon.createObserver(observerName)

	if condition == "for" then condition = args[3] end 
	if condition then 
		if not cfxZones.verifyMethod(condition, theZone) then 
			debugger.outText("*** new: illegal trigger condition <" .. condition .. "> for observer <" .. observerName .. ">", 30)
			return false 
		end
		theZone.debugInputMethod = condition
	end	
	
	debugger.addDebugger(theZone)
	debugger.outText("*** [" .. dcsCommon.nowString() .. "] debugger: new observer <" .. observerName .. "> for <" .. theZone.debugInputMethod .. ">", 30)
	return true 
end

function debugDemon.processUpdateCommand(args, event)
	-- syntax update <observername> [to] <condition>
	local observerName = args[1]
	if not observerName then 
		debugger.outText("*** update: missing observer name.", 30)
		return false -- allows correction 
	end
	
	-- see if this observer already existst
	local theZone = debugger.getDebuggerByName(observerName)
	if not theZone then 
		debugger.outText("*** update: observer <" .. observerName .. "> does not exist exists.", 30)
		return false -- allows correction 
	end
		
	local condition = args[2] -- may need remainder instead
	if condition == "to" then condition = args[3] end 
	if condition then 
		if not cfxZones.verifyMethod(condition, theZone) then 
			debugger.outText("*** update: illegal trigger condition <" .. condition .. "> for observer <" .. observerName .. ">", 30)
			return false 
		end
		theZone.debugInputMethod = condition
	end
	
	debugger.outText("*** [" .. dcsCommon.nowString() .. "] debugger: updated observer <" .. observerName .. "> to <" .. theZone.debugInputMethod .. ">", 30)
	return true 
end

function debugDemon.processDropCommand(args, event)
	-- syntax drop <observername>
	local observerName = event.remainder -- remainder
	if not observerName then 
		debugger.outText("*** drop: missing observer name.", 30)
		return false -- allows correction 
	end
	
	-- see if this observer already existst
	local theZone = debugger.getDebuggerByName(observerName)
	if not theZone then 
		debugger.outText("*** drop: observer <" .. observerName .. "> does not exist exists.", 30)
		return false -- allows correction 
	end
	
	-- now simply and irrevocable remove the observer, unless it's home, 
	-- in which case it's simply reset 
	if theZone == debugDemon.observer then 
		debugger.outText("*** drop: <" .. observerName .. "> is MY PRECIOUS and WILL NOT be dropped.", 30)
		-- can't really happen since it contains blanks, but
		-- we've seen stranger things
		return false -- allows correction 
	end 
	
	debugger.removeDebugger(theZone)
	
	debugger.outText("*** [" .. dcsCommon.nowString() .. "] debugger: dropped observer <" .. observerName .. ">", 30)
	return true 
end
-- observe command: add a new flag to observe
function debugDemon.processObserveCommand(args, event)
	-- syntax: observe <flagname> [with <observername>]
	-- args[1] is the name of the flag 
	local flagName = args[1]
	if not flagName then 
		debugger.outText("*** observe: missing flag name.", 30)
		return false -- allows correction 
	end
	
	local withTracker = nil 
	if args[2] == "with" then 
		local aName = args[3]
		if not aName then 
			debugger.outText("*** observe: missing <observer name> after 'with'.", 30)
			return false -- allows correction 
		end
		aName = dcsCommon.stringRemainsStartingWith(event.remainder, aName)
		withTracker = debugger.getDebuggerByName(aName)
		if not withTracker then 
--			withTracker = debugDemon.createObserver(aName)
--			debugger.addDebugger(withTracker)
			debugger.outText("*** observe: no observer <" .. aName .. "> exists", 30)
			return false -- allows correction
		end 
	else -- not with as arg 2 
		if #args > 1 then 
			debugger.outText("*** observe: unknown command after flag name '" .. flagName .. "'.", 30)
			return false -- allows correction 
		end
		-- use own observer  
		withTracker = debugDemon.observer
	end
	
	if debugger.isObservingWithObserver(flagName, withTracker) then 
		debugger.outText("*** observe: already observing " .. flagName .. " with <" .. withTracker.name .. ">" , 30)
		return true
	end
	
	-- we add flag to tracker and init value
	debugger.addFlagToObserver(flagName, withTracker)
	debugger.outText("*** [" .. dcsCommon.nowString() .. "] debugger: now observing <" .. flagName .. "> for value " .. withTracker.debugInputMethod .. " with <" .. withTracker.name .. ">.", 30)
	return true
end

function debugDemon.processShowCommand(args, event)
	-- syntax -show <name> with name either flag or observer
	-- observer has precendce over flag 
	local theName = args[1]
	if not theName then 
		debugger.outText("*** show: missing observer/flag name.", 30)
		return false -- allows correction 
	end
	
	-- now see if we have an observer 
	theName = dcsCommon.stringRemainsStartingWith(event.remainder, theName)
	local theObserver = debugger.getDebuggerByName(theName)
	
	if not theObserver then 
		-- we directly use trigger.misc
		local fVal = trigger.misc.getUserFlag(theName)
		debugger.outText("[" .. dcsCommon.nowString() .. "] flag <" .. theName .. "> : value <".. fVal .. ">", 30)
		return true 
	end
	
	-- if we get here, we want to show an entire observer 
	debugger.outText("*** [" .. dcsCommon.nowString() .. "] flags observed by <" .. theName .. "> looking for <value ".. theObserver.debugInputMethod .. ">:", 30)
	local flags = theObserver.flagArray
	local values = theObserver.valueArray
	for idx, flagName in pairs(flags) do 
		local lastVal = values[flagName]
		local fVal = cfxZones.getFlagValue(flagName, theObserver)
		-- add code to detect if it would trigger here 
		local hit = cfxZones.testFlagByMethodForZone(fVal, lastVal, theObserver.debugInputMethod, theObserver)
		local theMark = "   "
		local trailer = ""
		if hit then 
			theMark = " ! "
			trailer = ", HIT!"
		end 
		debugger.outText(theMark .. "f:<" .. flagName .. "> = <".. fVal .. "> [current, state = <" .. values[flagName] .. ">" .. trailer .. "]", 30)
	end 
	
	return true 
end

function debugDemon.createSnapshot(allObservers)
	if not allObservers then allObservers = {debugDemon.observer} end
	local snapshot = {}
	for idx, theZone in pairs(allObservers) do 

		-- iterate each observer 
		for idy, flagName in pairs (theZone.flagArray) do 
			local fullName = cfxZones.expandFlagName(flagName, theZone)
			local fVal = trigger.misc.getUserFlag(fullName)
			snapshot[fullName] = fVal
		end
	end
	return snapshot
end

function debugDemon.processSnapCommand(args, event)
-- syntax snap [<observername>]
	local allObservers = debugger.debugZones -- default: all zones
	local theObserver = nil 
	
	local theName = args[1]
	if theName then 
		-- now see if we have an observer 
		theName = dcsCommon.stringRemainsStartingWith(event.remainder, theName)
		theObserver = debugger.getDebuggerByName(theName)
		if not theObserver then 
			debugger.outText("*** snap: unknown observer name <" .. theName .. ">.", 30)
			return false -- allows correction
		end
	end 
	
	if theObserver then 
		allObservers = {}
		table.insert(allObservers, theObserver)
	end
	
	-- set up snapshot 
	local snapshot = debugDemon.createSnapshot(allObservers) 
	
	local sz = dcsCommon.getSizeOfTable(snapshot)
	debugDemon.snapshot = snapshot
	debugger.outText("*** [" .. dcsCommon.nowString() .. "] debug: new snapshot created, " .. sz .. " flags.", 30)
	
	return true 
end

function debugDemon.processCompareCommand(args, event)
	debugger.outText("*** [" .. dcsCommon.nowString() .. "] debug: comparing snapshot with current flag values", 30)
	for flagName, val in pairs (debugDemon.snapshot) do 
		local cVal = trigger.misc.getUserFlag(flagName)
		local mark = '   '
		if cVal ~= val then mark = ' ! ' end
		debugger.outText(mark .. "<" .. flagName .. "> snap = <" .. val .. ">, now = <" .. cVal .. "> " .. mark, 30)
	end
	debugger.outText("*** END", 30)
	return true 
end

function debugDemon.processNoteCommand(args, event)
	local n = event.remainder
	debugger.outText("*** [" .. dcsCommon.nowString() .. "]: " .. n, 30)
	return true
end

function debugDemon.processSetCommand(args, event)
	-- syntax set <flagname> <value>
	local theName = args[1]
	if not theName then 
		debugger.outText("*** set: missing flag name.", 30)
		return false -- allows correction 
	end
	
	local theVal = args[2]
	if theVal and type(theVal) == "string" then 
		theVal = theVal:upper()
		if theVal == "YES" or theVal == "TRUE" then theVal = "1" end
		if theVal == "NO" or theVal == "FALSE" then theVal = "0" end
	end
	
	if not theVal or not (tonumber(theVal)) then 
		debugger.outText("*** set: missing or illegal value for flag <" .. theName .. ">.", 30)
		return false -- allows correction
	end 
	
	theVal = tonumber(theVal) 
	trigger.action.setUserFlag(theName, theVal)
	-- we set directly, no cfxZones proccing
	local note =""
	-- flags are ints only?
	if theVal ~= math.floor(theVal) then 
		note = " [int! " .. math.floor(theVal) .. "]"
	end
	
	debugger.outText("*** [" .. dcsCommon.nowString() .. "] debug: set flag <" .. theName .. "> to <" .. theVal .. ">" .. note, 30)
	
	local newVal = trigger.misc.getUserFlag(theName)
	if theVal ~= newVal then 
		debugger.outText("*** [" .. dcsCommon.nowString() .. "] debug: readback failure for flag <" .. theName .. ">: expected <" .. theVal .. ">, got <" .. newVal .. "!", 30)
	end 
	
	return true 
end

function debugDemon.processIncCommand(args, event)
	-- syntax inc <flagname>
	local theName = args[1]
	if not theName then 
		debugger.outText("*** inc: missing flag name.", 30)
		return false -- allows correction 
	end
	
	local cVal = trigger.misc.getUserFlag(theName)
	local nVal = cVal + 1 
	
	-- we set directly, no cfxZones procing
	debugger.outText("*** [" .. dcsCommon.nowString() .. "] debug: inc flag <" .. theName .. "> from <" .. cVal .. "> to <" .. nVal .. ">", 30)
	trigger.action.setUserFlag(theName, nVal)
	return true 
end

function debugDemon.processFlipCommand(args, event)
	-- syntax flip <flagname> 
	local theName = args[1]
	if not theName then 
		debugger.outText("*** flip: missing flag name.", 30)
		return false -- allows correction 
	end
	
	local cVal = trigger.misc.getUserFlag(theName)
	if cVal == 0 then nVal = 1 else nVal = 0 end 
	
	-- we set directly, no cfxZones procing
	debugger.outText("*** [" .. dcsCommon.nowString() .. "] debug: flipped flag <" .. theName .. "> from <" .. cVal .. "> to <" .. nVal .. ">", 30)
	trigger.action.setUserFlag(theName, nVal)
	return true 
end

function debugDemon.processListCommand(args, event)
	-- syntax list or list <prefix>
	local prefix = nil 
	prefix = args[1]
	if prefix then 
		prefix = event.remainder -- dcsCommon.stringRemainsStartingWith(event.text, prefix)
	end
	if prefix then 
		debugger.outText("*** [" .. dcsCommon.nowString() .. "] listing observers whose name contains <" .. prefix .. ">:", 30)
	else 
		debugger.outText("*** [" .. dcsCommon.nowString() .. "] listing all observers:", 30)
	end
	
	local allObservers = debugger.debugZones
	for idx, theZone in pairs(allObservers) do 
		local theName = theZone.name 
		local doList = true 
		if prefix then 
			doList = dcsCommon.containsString(theName, prefix, false)
		end
		
		if doList then 
			debugger.outText("  <" .. theName .. "> for <value " .. theZone.debugInputMethod .. "> (" .. #theZone.flagArray .. " flags)", 30)
		end
	end
    return true 
end

function debugDemon.processWhoCommand(args, event)
	-- syntax: who <flagname>
	local flagName = event.remainder -- args[1]
	if not flagName or flagName:len()<1 then 
		debugger.outText("*** who: missing flag name.", 30)
		return false -- allows correction 
	end

	local observers = debugger.isObserving(flagName)

	if not observers or #observers < 1 then 
		debugger.outText("*** [" .. dcsCommon.nowString() .. "] flag <" .. flagName .. "> is currently not observed", 30)
		return false
	end 

	debugger.outText("*** [" .. dcsCommon.nowString() .. "] flag <" .. flagName .. "> is currently observed by", 30)
	for idx, theZone in pairs(observers) do 
		debugger.outText("  <" .. theZone.name .. "> looking for <value " .. theZone.debugInputMethod .. ">", 30)
	end
	
	return true
	
end

function debugDemon.processForgetCommand(args, event)
	-- syntax: forget <flagname> [with <observername>]

	local flagName = args[1]
	if not flagName then 
		debugger.outText("*** forget: missing flag name.", 30)
		return false -- allows correction 
	end
	
	local withTracker = nil 
	if args[2] == "with" or args[2] == "from" then -- we also allow 'from'
		local aName = args[3]
		if not aName then 
			debugger.outText("*** forget: missing <observer name> after 'with'.", 30)
			return false -- allows correction 
		end
		
		aName = dcsCommon.stringRemainsStartingWith(event.remainder, aName)
		withTracker = debugger.getDebuggerByName(aName)
		if not withTracker then 
			debugger.outText("*** forget: no observer named <" .. aName .. ">", 30)
			return false
		end 
	else -- not with as arg 2 
		if #args > 1 then 
			debugger.outText("*** forget: unknown command after flag name '" .. flagName .. "'.", 30)
			return false -- allows correction 
		end
		-- use own observer  
		withTracker = debugDemon.observer
	end
	
	if not debugger.isObservingWithObserver(flagName, withTracker) then 
		debugger.outText("*** forget: observer <" .. withTracker.name .. "> does not observe flag <" .. flagName .. ">", 30)
		return false
	end
	
	-- we add flag to tracker and init value
	debugger.removeFlagFromObserver(flagName, withTracker)
	debugger.outText("*** [" .. dcsCommon.nowString() .. "] debugger: no longer observing " .. flagName .. " with <" .. withTracker.name .. ">.", 30)
	return true
end


function debugDemon.processStartCommand(args, event)
	debugger.doActivate()
	return true 
end

function debugDemon.processStopCommand(args, event)
	debugger.doDeactivate()
	return true 
end

function debugDemon.processResetCommand(args, event)
	-- supports reset <observer> 
	-- syntax: forget <flagname> [with <observername>]

	local obsName = args[1]
	if not obsName then 
		debugger.reset() -- reset all
		debugger.outText("*** [" .. dcsCommon.nowString() .. "] debug: reset complete.", 30)
		return true -- allows correction 
	end
	
	local withTracker = nil 
	local aName = event.remainder 
	withTracker = debugger.getDebuggerByName(aName)
	if not withTracker then 
		debugger.outText("*** reset: no observer <" .. aName .. ">", 30)
		return false
	end 
	
	debugger.resetObserver(withTracker)
	
	debugger.outText("*** [" .. dcsCommon.nowString() .. "] debugger:reset observer <" .. withTracker.name .. ">", 30)
	return true
end

function debugDemon.processSaveCommand(args, event)
	-- save log to file, requires persistence module 
	-- syntax: -save [<fileName>]
	local aName = event.remainder
	if not aName or aName:len() < 1 then 
		aName = "DML Debugger Log"
	end
	if not dcsCommon.stringEndsWith(aName, ".txt") then 
		aName = aName .. ".txt"
	end
	debugger.saveLog(aName)
	return true 
end

function debugDemon.processRemoveCommand(args, event)
	-- remove a group, unit or object 
	-- try group first 
	local aName = event.remainder
	if not aName or aName:len() < 1 then 
		debugger.outText("*** remove: no remove target", 30)
		return false
	end
	
	aName = dcsCommon.trim(aName)
	local theGroup = Group.getByName(aName)
	if theGroup and theGroup:isExist() then 
		theGroup:destroy()
		debugger.outText("*** remove: removed group <" .. aName .. ">", 30)
		return true
	end
	
	local theUnit = Unit.getByName(aName)
	if theUnit and theUnit:isExist() then 
		theUnit:destroy()
		debugger.outText("*** remove: removed unit <" .. aName .. ">", 30)
		return true
	end
	
	local theStatic = StaticObject.getByName(aName)
	if theStatic and theStatic:isExist() then 
		theStatic:destroy()
		debugger.outText("*** remove: removed static object <" .. aName .. ">", 30)
		return true
	end
	debugger.outText("*** remove: did not find anything called <" .. aName .. "> to remove", 30)
		return true
end

function debugDemon.doEventMon(theEvent)
	if not theEvent then return end 
	if not debugger.active then return end 
	local ID = theEvent.id 
	if debugger.showEvents[ID] then
		-- we show this event 
		m = "*** event <" .. debugger.showEvents[ID] .. ">"
		-- see if we have initiator
		if theEvent.initiator then 
			local theUnit = theEvent.initiator
			if Unit.isExist(theUnit) then 
				m = m .. " for "
				if theUnit.getPlayerName and theUnit:getPlayerName() then 
					m = m .. "player = " .. theUnit:getPlayerName() .. " in "
				end
				m = m .. "unit <" .. theUnit:getName() .. ">"
			end 
		end 
		debugger.outText(m, 30)
		-- save it to lastevent so we can analyse 
		debugger.lastEvent = theEvent 
	end
end 

debugDemon.m = ""
-- dumpVar2m, invoke externally dumpVar2m(varname, var)
function debugDemon.dumpVar2m(key, value, prefix, inrecursion)
	-- based on common's dumpVar, appends to var "m" 
	if not inrecursion then 
		-- start, init m
		debugDemon.m = "analysis of <" .. key .. ">\n==="
	end
	if not value then value = "nil" end
	if not prefix then prefix = "" end
	prefix = " " .. prefix
	if type(value) == "table" then 
		debugDemon.m = debugDemon.m .. "\n" .. prefix .. key .. ": [ "
		-- iterate through all kvp
		for k,v in pairs (value) do
			debugDemon.dumpVar2m(k, v, prefix, true)
		end
		debugDemon.m = debugDemon.m .. "\n" .. prefix .. " ] - end " .. key
		
	elseif type(value) == "boolean" then 
		local b = "false"
		if value then b = "true" end
		debugDemon.m = debugDemon.m .. "\n" .. prefix .. key .. ": " .. b

	else -- simple var, show contents, ends recursion
		debugDemon.m = debugDemon.m .. "\n" .. prefix .. key .. ": " .. value
	end
	
	if not inrecursion then 
		-- output a marker to find in the log / screen
		debugDemon.m = debugDemon.m .. "\n" .. "=== analysis end\n"
	end
end

function debugDemon.processEventMonCommand(args, event)
	-- turn event monitor all/off/?/last  
	-- syntax: -eventmon  on|off 
	local aParam = dcsCommon.trim(event.remainder)
	if not aParam or aParam:len() < 1 then 
		aParam = "all"
	end
	aParam = string.upper(aParam)
	evtNum = tonumber(aParam)
	if aParam == "ON" or aParam == "ALL" then 
		debugger.outText("*** eventmon: turned ON, showing ALL events", 30)
		local events = {}
		for idx,evt in pairs(debugDemon.eventList) do 
			events[tonumber(idx)] = evt
		end 
		debugger.showEvents = events 
	elseif evtNum then -- add the numbered to 
		debugger.eventmon = false 
		if evtNum <= 0 then evtNum = 0 end 
		if evtNum >= 57 then evtNum = 35 end 
		debugger.showEvents[evtNum] = debugDemon.eventList[tostring(evtNum)] 
		debugger.outText("*** eventmon: added event <" .. debugger.showEvents[evtNum] .. ">", 30)
	elseif aParam == "OFF" then 
		debugger.showEvents = {}
		debugger.outText("*** eventmon: removed all events from monitor list", 30)
	elseif aParam == "?" then 
		local m = "*** eventmon: currently tracking these events:"
		for idx, evt in pairs(debugger.showEvents) do 
			m = m .. "\n" ..  evt 
		end			
		debugger.outText(m .. "\n*** end of list", 30)
	elseif aParam == "LAST" then 
		if debugger.lastEvent then 
			debugDemon.dumpVar2m("event", debugger.lastEvent)
			debugger.outText(debugDemon.m, 39)
		else 
			debugger.outText("*** eventmon: no event on record", 39)
		end
	else 
		debugger.outText("*** eventmon: unknown parameter <" .. event.remainder .. ">", 30)
	end
	return true 
end 

--
-- read and write directly to Lua tables
--

function debugDemon.processQueryCommand(args, event)
	-- syntax -q <name> with name a (qualified) Lua table reference 
	local theName = args[1]

	if not theName then 
		debugger.outText("*** q: missing Lua table/element name.", 30)
		return false -- allows correction 
	end
	theName = dcsCommon.stringRemainsStartingWith(event.remainder, theName)

	-- put this into a string, and execute it 
	local exec = "return " .. theName 
	local f = loadstring(exec) 
	local res
	if pcall(f) then 
		res = f()
		if type(res) == "boolean" then 
			res = "[BOOL FALSE]"
			if res then res = "[BOOL TRUE]" end 
		elseif type(res) == "table" then res = "[Lua Table]"
		elseif type(res) == "nil" then res = "[NIL]"
		elseif type(res) == "function" then res = "[Lua Function]"
		elseif type(res) == "number" or type(res) == "string" then 
			res = res .. " (a " .. type(res) .. ")"
		else res = "[Lua " .. type(res) .. "]"
		end
	else 
		res = "[Lua error]"
	end 
	
	debugger.outText("[" .. dcsCommon.nowString() .. "] <" .. theName .. "> = ".. res, 30)
	
	return true 
end

function debugDemon.processAnalyzeCommand(args, event)
	-- syntax -a <name> with name a (qualified) Lua table reference 
	local theName = args[1]

	if not theName then 
		debugger.outText("*** a: missing Lua table/element name.", 30)
		return false -- allows correction 
	end
	theName = dcsCommon.stringRemainsStartingWith(event.remainder, theName)

	-- put this into a string, and execute it 
	local exec = "return " .. theName 
	local f = loadstring(exec) 
	local res
	if pcall(f) then 
		res = f()
		debugDemon.dumpVar2m(theName, res)
		res = debugDemon.m
	else 
		res = "[Lua error]"
	end 
	
	debugger.outText("[" .. dcsCommon.nowString() .. "] <" .. theName .. "> = ".. res, 30)
	
	return true 
end

function debugDemon.processWriteCommand(args, event)
	-- syntax -w <name> <value> with name a (qualified) Lua table reference and value a Lua value (including strings, with quotes of course). {} means an empty set etc. you CAN call into DCS MSE with this, and create a lot of havoc.
	-- also, allow "=" semantic, -w p = {x=1, y=2}
	
	local theName = args[1]
	if not theName then 
		debugger.outText("*** w: missing Lua table/element name.", 30)
		return false -- allows correction 
	end
	local param = args [2]
	if param == "=" then param = args[3] end 
	if not param then 
		debugger.outText("*** w: missing value to set to")
		return false 
	end 

	param = dcsCommon.stringRemainsStartingWith(event.remainder, param)

	-- put this into a string, and execute it 
	local exec = theName .. " = " .. param 
	local f = loadstring(exec) 
	local res
	if pcall(f) then 
		res = "<" .. theName .. "> set to <" .. param .. ">"
	else 
		res = "[Unable to set - Lua error]"
	end 
	
	debugger.outText("[" .. dcsCommon.nowString() .. "] " .. res, 30)
	
	return true 
end

--
-- smoke & boom 
-- 

function debugDemon.processSmokeCommand(args, event)
	-- syntax -color 
	local color = 0 -- green default 
	local colorCom = args[1]
	if colorCom then 
		colorCom = colorCom:lower()
		if colorCom == "red" or colorCom == "1" then color = 1
		elseif colorCom == "white" or colorCom == "2" then color = 2 
		elseif colorCom == "orange" or colorCom == "3" then color = 3 
		elseif colorCom == "blue" or colorCom == "4" then color = 4
		elseif colorCom == "green" or colorCom == "0" then color = 0
		else 
			debugger.outText("*** smoke: unknown color <" .. colorCom .. ">, using green.", 30)
		end 
		local pos = event.pos 
		local h = land.getHeight({x = pos.x, y = pos.z}) + 1
		local p = { x = event.pos.x, y = h, z = event.pos.z} 
		trigger.action.smoke(p, color)
		debugger.outText("*** smoke: placed smoke at <" .. dcsCommon.point2text(p, true) .. ">.", 30)
	end 
end 

function debugDemon.processBoomCommand(args, event)
	-- syntax -color 
	local power = 1 -- boom default 
	local powerCom = args[1]
	if powerCom then 
		powerCom = tonumber(powerCom)
		if powerCom then
			power = powerCom
		end 
	end
	local pos = event.pos 
	local h = land.getHeight({x = pos.x, y = pos.z}) + 1
	local p = { x = event.pos.x, y = h, z = event.pos.z} 
	trigger.action.explosion(p, power)
	debugger.outText("*** boom: placed <" .. power .. "> explosion at <" .. dcsCommon.point2text(p, true) .. ">.", 30) 
end 

--
-- spawning units at the location of the mark 
--

function debugDemon.getCoaFromCommand(args)
	for i=1, #args do
		local aParam = args[i]
		if dcsCommon.stringStartsWith(aParam, "red", true) then return 1, i end
		if dcsCommon.stringStartsWith(aParam, "blu", true) then return 2, i end
		if dcsCommon.stringStartsWith(aParam, "neu", true) then return 0, i end
	end
	return 0, nil  
end 

function debugDemon.getAirFromCommand(args)
	for i=1, #args do
		local aParam = args[i]
		if aParam:lower() == "inair" then return true, i end
		if aParam:lower() == "air" then return true, i end
	end
	return false, nil  
end 

function debugDemon.getHeadingFromCommand(args)
	for i=1, #args do
		local aParam = args[i]
		if dcsCommon.stringStartsWith(aParam, "heading=", true) then 
			local parts = dcsCommon.splitString(aParam, "=")
			local num = parts[2]
			if num and tonumber(num) then 
				return tonumber(num), i
			end 
		end
	end
	return 0, nil  
end

function debugDemon.getNumFromCommand(args)
	for i=1, #args do
		local aParam = args[i]
		local num = tonumber(aParam)
		if num then return num, i end
	end
	return 1, nil  
end 

function debugDemon.processSpawnCommand(args, event)
	-- complex syntax: 
	-- spawn [red|blue|neutral] [number] <type> [heading=<number>] | "?"
	local params = dcsCommon.clone(args)
--	for i=1, #params do 
--		trigger.action.outText("arg[" .. i .."] = <" .. params[i] .. ">", 30)
--	end
	
	-- get coalition from input 
	
	local coa, idx = debugDemon.getCoaFromCommand(params)
	if idx then table.remove(params, idx) end 
	local inAir, idy = debugDemon.getAirFromCommand(params)
	if idy then table.remove(params, idy) end
	local num, idz = debugDemon.getNumFromCommand(params)
	if idz then table.remove(params, idz) end 
	local heading, idk = debugDemon.getHeadingFromCommand(params)
	if idk then table.remove(params, idk) end 
	
	local class = params[1]
	if not class then 
		debugger.outText("*** spawn: missing keyword (what to spawn).", 30)
		return 
	end 
	
	class = class:lower() 
	
	-- when we are here, we have reduced all params, so class is [1]
--	trigger.action.outText("spawn with class <" .. class .. ">, num <" .. num .. ">, inAir <" .. dcsCommon.bool2Text(inAir) .. ">, coa <" .. coa .. ">, hdg <" .. heading .. ">", 30)
	heading = heading  * 0.0174533 -- in rad 
	
	local pos = event.pos 
	local h = land.getHeight({x = pos.x, y = pos.z}) + 1
	local p = { x = event.pos.x, y = h, z = event.pos.z} 
		
	if class == "tank" or class == "tanks" then 
		-- spawn the 'tank' class 
		local theType = debugger.spawnTypes["tank"]
		return debugDemon.spawnTypeWithCat(theType, coa, num, p,  nil,heading)
	elseif class == "man" or class == "soldier" or class == "men" then 
		local theType = debugger.spawnTypes["inf"]
		return debugDemon.spawnTypeWithCat(theType, coa, num, p,  nil,heading)
	
	elseif class == "inf" or class == "ifv" or class == "sam" or 
	       class == "arty" or class == "aaa" then 
		local theType = debugger.spawnTypes[class]
		return debugDemon.spawnTypeWithCat(theType, coa, num, p,  nil,heading)
	elseif class == "truck" or class == "trucks" then 
		local theType = debugger.spawnTypes["truck"]
		return debugDemon.spawnTypeWithCat(theType, coa, num, p,  nil,heading)	
	elseif class == "manpad" or class == "manpads" or class == "pad" or class == "pads" then 
		local theType = debugger.spawnTypes["manpad"]
		return debugDemon.spawnTypeWithCat(theType, coa, num, p,  nil,heading)

	elseif class == "ship" or class == "ships" then 
		local theType = debugger.spawnTypes["ship"]
		return debugDemon.spawnTypeWithCat(theType, coa, num, p, Group.Category.SHIP, heading)

	elseif class == "jet" or class == "jets" then 
		local theType = debugger.spawnTypes["jet"]
		return debugDemon.spawnAirWIthCat(theType, coa, num, p, nil, 1000, 160, heading)

	elseif class == "ww2" then 
		local theType = debugger.spawnTypes[class]
		return debugDemon.spawnAirWIthCat(theType, coa, num, p, nil, 1000, 100, heading)

	elseif class == "bomber" or class == "awacs" then 
		local theType = debugger.spawnTypes[class]
		return debugDemon.spawnAirWIthCat(theType, coa, num, p, nil, 8000, 200, heading)
		
	elseif class == "drone" then 
		local theType = debugger.spawnTypes[class]
		return debugDemon.spawnAirWIthCat(theType, coa, num, p, nil, 3000, 77, heading)
	
	elseif class == "helo" or class == "helos" then 
		local theType = debugger.spawnTypes["helo"]
		return debugDemon.spawnAirWIthCat(theType, coa, num, p, Group.Category.HELICOPTER, 200, 40, heading)
		
	elseif class == "cargo" or class == "obj" then 
		local isCargo = (class == "cargo") 
		local theType = debugger.spawnTypes[class]
		return debugDemon.spawnObjects(theType, coa, num, p, isCargo, heading)
	
	elseif class == "?" then 
		local m = " spawn: invoke '-spawn [number] [coalition] <type> [heading]' with \n" ..
		" number = any number, default is 1\n" ..
		" coalition = 'red' | 'blue' | 'neutral', default is neutral\n" ..
		" heading = 'heading=<number>' - direction to face, in degrees, no blanks\n" ..
		" <type> = what to spawn, any of the following pre-defined (no quotes)\n" ..
		"   'tank' - a tank " .. debugDemon.tellType("tank") .. "\n" ..
		"   'ifv' - an IFV " .. debugDemon.tellType("ifv") .. "\n" ..
		"   'inf' - an infantry soldier " .. debugDemon.tellType("inf") .. "\n" ..
		"   'sam' - a SAM vehicle " .. debugDemon.tellType("sam") .. "\n" .. 
		"   'aaa' - a AAA vehicle " .. debugDemon.tellType("aaa") .. "\n" ..
		"   'arty' - artillery vehicle " .. debugDemon.tellType("arty") .. "\n" ..
		"   'manpad' - a soldier with SAM " .. debugDemon.tellType("manpad") .. "\n" ..
		"   'truck' - a truck " .. debugDemon.tellType("truck") .. "\n\n" ..
		"   'jet' - a fast aircraft " .. debugDemon.tellType("jet") .. "\n" ..
		"   'ww2' - a warbird " .. debugDemon.tellType("ww2") .. "\n" ..
		"   'bomber' - a heavy bomber " .. debugDemon.tellType("bomber") .. "\n" ..
		"   'awacs' - an AWACS plane " .. debugDemon.tellType("awacs") .. "\n" ..
		"   'drone' - a drone " .. debugDemon.tellType("drone") .. "\n" ..
		"   'helo' - a helicopter " .. debugDemon.tellType("helo") .. "\n\n" ..
		"   'ship' - a naval unit" .. debugDemon.tellType("ship") .. "\n\n" ..
		"   'cargo' - some helicopter cargo " .. debugDemon.tellType("cargo") .. "\n" ..
		"   'obj' - a static object " .. debugDemon.tellType("obj") .. "\n" 
		
		debugger.outText(m, 30)
		return true 
	else 
		debugger.outText("*** spawn: unknown kind <" .. class .. ">.", 30)
		return false 
	end 
end

function debugDemon.tellType(theType)
	return " [" .. debugger.spawnTypes[theType] .. "]"
end 

function debugDemon.spawnTypeWithCat(theType, coa, num, p, cat, heading)
	trigger.action.outText("heading is <" .. heading .. ">", 30)
	if not cat then cat = Group.Category.GROUND end 
	if not heading then heading = 0 end 
	
	local xOff = 0
	local yOff = 0
	-- build group 
	local groupName = dcsCommon.uuid(theType)
	local gData = dcsCommon.createEmptyGroundGroupData(groupName)
	for i=1, num do 
		local aUnit = {}
		aUnit = dcsCommon.createGroundUnitData(groupName .. "-" .. i, theType)
		--aUnit.heading = heading 
		dcsCommon.addUnitToGroupData(aUnit, gData, xOff, yOff, heading)
		xOff = xOff + 10 
		yOff = yOff + 10
	end 
	
	-- arrange in a grid formation
	local radius = math.floor(math.sqrt(num) * 10)
	if cat == Group.Category.SHIP then 
		radius = math.floor(math.sqrt(num) * 100)
	end 
	
	dcsCommon.arrangeGroupDataIntoFormation(gData, radius, 10, "GRID")
	
	-- move to destination 
	dcsCommon.moveGroupDataTo(gData, p.x, p.z)

	-- spawn 
	local cty = dcsCommon.getACountryForCoalition(coa)
	local theGroup = coalition.addGroup(cty, cat, gData)
	if theGroup then 
		debugger.outText("[" .. dcsCommon.nowString() .. "] created units at " .. dcsCommon.point2text(p, true), 30)
		return true
	else 
		debugger.outText("[" .. dcsCommon.nowString() .. "] failed to created units", 30)
		return false
	end 
	return false
end 

function debugDemon.spawnAirWIthCat(theType, coa, num, p, cat, alt, speed, heading)
	if not cat then cat = Group.Category.AIRPLANE end 
	local xOff = 0
	local yOff = 0
	-- build group 
	local groupName = dcsCommon.uuid(theType)
	local gData = dcsCommon.createEmptyAircraftGroupData(groupName)
	for i=1, num do 
		local aUnit = {}
		aUnit = dcsCommon.createAircraftUnitData(groupName .. "-" .. i, theType, false, alt, speed)
		--aUnit.heading = heading 
		dcsCommon.addUnitToGroupData(aUnit, gData, xOff, yOff, heading)
		xOff = xOff + 30 
		yOff = yOff + 30
	end 
	-- move to destination 
	dcsCommon.moveGroupDataTo(gData, p.x, p.z)

	-- make waypoints: initial point and 200 km away in direction heading
	local p2 = dcsCommon.pointInDirectionOfPointXYY(heading, 200000, p)
	local wp1 = dcsCommon.createSimpleRoutePointData(p, alt, speed)
	local wp2 = dcsCommon.createSimpleRoutePointData(p2, alt, speed)
	-- add waypoints 
	dcsCommon.addRoutePointForGroupData(gData, wp1)
	dcsCommon.addRoutePointForGroupData(gData, wp2)
	
	-- spawn 
	local cty = dcsCommon.getACountryForCoalition(coa)
	local theGroup = coalition.addGroup(cty, cat, gData)
	if theGroup then 
		debugger.outText("[" .. dcsCommon.nowString() .. "] created air units at " .. dcsCommon.point2text(p, true), 30)
		return true
	else 
		debugger.outText("[" .. dcsCommon.nowString() .. "] failed to created air units", 30)
		return false
	end 
end

function debugDemon.spawnObjects(theType, coa, num, p, cargo, heading)
	if not cargo then cargo = false end 
	local cty = dcsCommon.getACountryForCoalition(coa)
	local xOff = 0
	local yOff = 0
	local success = false 
	-- build static objects and spawn individually
	for i=1, num do 
		local groupName = dcsCommon.uuid(theType)
		local gData = dcsCommon.createStaticObjectData(groupName, theType, 0, false, cargo, 1000)
		gData.x = xOff + p.x 
		gData.y = yOff + p.z 
		gData.heading = heading
		local theGroup = coalition.addStaticObject(cty, gData)
		success = theGroup
		xOff = xOff + 10 -- stagger by 10m, 10m 
		yOff = yOff + 10
	end 
	
	-- was it worth it?
	if success then 
		debugger.outText("[" .. dcsCommon.nowString() .. "] created objects at " .. dcsCommon.point2text(p, true), 30)
		return true
	else 
		debugger.outText("[" .. dcsCommon.nowString() .. "] failed to create objects", 30)
		return false
	end 
end 


--
-- init and start
--

function debugDemon.readConfigZone()
	local theZone = cfxZones.getZoneByName("debugDemonConfig") 
	if not theZone then 
		if debugDemon.verbose then 
			debugger.outText("+++debug (daemon): NO config zone!", 30)
		end 
		theZone = cfxZones.createSimpleZone("debugDemonConfig") 
	end 
	debugDemon.configZone = theZone 
	
	debugDemon.keepOpen = cfxZones.getBoolFromZoneProperty(theZone, "keepOpen", false)
	
	debugDemon.markOfDemon = cfxZones.getStringFromZoneProperty(theZone,"mark", "-") -- all commands must start with this sequence

	
	debugDemon.verbose = cfxZones.getBoolFromZoneProperty(theZone, "verbose", false)
	
	
	if debugger.verbose then 
		debugger.outText("+++debug (deamon): read config", 30)
	end 
end


function debugDemon.init()
	if not dcsCommon.libCheck then 
		trigger.action.outText("cfx interactive debugger requires dcsCommon", 30)
		return false 
	end 
	if not dcsCommon.libCheck("cfx interactive debugger", debugDemon.requiredLibs) then
		return false 
	end
	
	-- config
	debugDemon.readConfigZone()
	
	-- now add known commands to interpreter. 
	debugDemon.addCommndProcessor("observe", debugDemon.processObserveCommand)
	debugDemon.addCommndProcessor("o", debugDemon.processObserveCommand) -- synonym

	debugDemon.addCommndProcessor("forget", debugDemon.processForgetCommand)

	debugDemon.addCommndProcessor("show", debugDemon.processShowCommand)
	debugDemon.addCommndProcessor("set", debugDemon.processSetCommand)
	debugDemon.addCommndProcessor("inc", debugDemon.processIncCommand)
	debugDemon.addCommndProcessor("flip", debugDemon.processFlipCommand)
    debugDemon.addCommndProcessor("list", debugDemon.processListCommand)
	debugDemon.addCommndProcessor("who", debugDemon.processWhoCommand)
	
	debugDemon.addCommndProcessor("new", debugDemon.processNewCommand)
	debugDemon.addCommndProcessor("update", debugDemon.processUpdateCommand)
	debugDemon.addCommndProcessor("drop", debugDemon.processDropCommand)
	
	debugDemon.addCommndProcessor("snap", debugDemon.processSnapCommand)
	debugDemon.addCommndProcessor("compare", debugDemon.processCompareCommand)
	debugDemon.addCommndProcessor("note", debugDemon.processNoteCommand)
	
	debugDemon.addCommndProcessor("start", debugDemon.processStartCommand)
	debugDemon.addCommndProcessor("stop", debugDemon.processStopCommand)
	debugDemon.addCommndProcessor("reset", debugDemon.processResetCommand)
	
	debugDemon.addCommndProcessor("save", debugDemon.processSaveCommand)

	debugDemon.addCommndProcessor("?", debugDemon.processHelpCommand)
	debugDemon.addCommndProcessor("help", debugDemon.processHelpCommand)

	debugDemon.addCommndProcessor("remove", debugDemon.processRemoveCommand)
	debugDemon.addCommndProcessor("spawn", debugDemon.processSpawnCommand)
	debugDemon.addCommndProcessor("add", debugDemon.processSpawnCommand)

	debugDemon.addCommndProcessor("eventmon", debugDemon.processEventMonCommand)
	debugDemon.addCommndProcessor("q", debugDemon.processQueryCommand)
	debugDemon.addCommndProcessor("w", debugDemon.processWriteCommand)
	debugDemon.addCommndProcessor("a", debugDemon.processAnalyzeCommand)
	debugDemon.addCommndProcessor("smoke", debugDemon.processSmokeCommand)
	debugDemon.addCommndProcessor("boom", debugDemon.processBoomCommand)
	return true 
end

function debugDemon.start()
	-- add my own debug zones to debugger so it can 
	-- track any changes 
	
	local observer = debugDemon.createObserver(debugDemon.myObserverName)
	debugDemon.observer = observer
	debugger.addDebugger(observer)
	
	-- create initial snapshot 
	debugDemon.snapshot = debugDemon.createSnapshot(debugger.debugZones)
	debugDemon.demonID = world.addEventHandler(debugDemon)
		
	debugger.outText("interactive debugDemon v" .. debugDemon.version .. " started" .. "\n  enter " .. debugDemon.markOfDemon .. "? in a map mark for help", 30)
	
	if not _G["persistence"] then 
		debugger.outText("\n  note: '-save' disabled, no persistence module found", 30)
	end
end

if debugDemon.init() then 
	debugDemon.start()
else 
	trigger.action.outText("*** interactive debugger failed to initialize.", 30)
	debugDemon = {}
end

--[[--
	- track units/groups/objects: health changes 
	- track players: unit change, enter, exit 
	- inspect objects, dumping category, life, if it's tasking, latLon, alt, speed, direction 
	
	- exec files. save all commands and then run them from script 
	
	- xref: which zones/attributes reference a flag, g.g. '-xref go'
		
	- track lua vars for change in value
	
--]]--
